#define EGLDISPLAY_VERSION "version 0.2"
char *egldisplayVersion = EGLDISPLAY_VERSION;

//#define DEBUG

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <EGL/egl.h>
#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#ifdef USE_IMLIB2
#include <Imlib2.h>
#endif /* USE_IMLIB2 */

void closeWindow(void);
void eventLoop(void);
GLuint loadProgram(const char *vertex_src, const char *fragment_src);
GLuint loadShader(const char *shader_source, GLenum type);
GLubyte *readImage(char *path, int *width, int *height,
                   int *tex_width, int *tex_height);
void redrawWindow(void);
void usageHelp(char *cmd);

#ifdef PANDORA
/* workaround ? */
float log2f(float x);
#endif /* PANDORA */

/* X11 */
Display *xDisplay = NULL;
Window xWindow = 0;
Atom wmDeleteWindow;
int winWidth;
int winHeight;

/* OpenGL ES2 */
EGLDisplay eglDisplay = EGL_NO_DISPLAY;
EGLContext eglContext = EGL_NO_CONTEXT;
EGLSurface eglSurface = EGL_NO_SURFACE;

/* Back Ground Texture */
GLuint bgProgram = 0;
GLuint bgTexture = 0;
GLint bgPositionLoc;
GLint bgTexCoordLoc;
GLint bgTextureLoc;
GLfloat bgTexVertices[] = {  0.0f,  0.0f,  // TexCoord 0
                             0.0f,  1.0f,  // TexCoord 1
                             1.0f,  1.0f,  // TexCoord 2
                             1.0f,  0.0f   // TexCoord 3
                          };

/* Command Option */
char *cmdName;

GLubyte *readImage(char *path, int *width, int *height,
                   int *tex_width, int *tex_height)
{
  GLubyte *pixels = NULL;

#ifdef USE_IMLIB2
  int i;
  int x;
  int y;
  int log_width;
  int log_height;
  GLubyte rv;
  GLubyte gv;
  GLubyte bv;
  DATA32 *image_data;
  Imlib_Image imlib_image;

  union i_to_c {
    DATA32 data_int;
    GLubyte data_char[4];
  } itoc;

  imlib_image = imlib_load_image(path);
  if(imlib_image == NULL){
    printf("readImage: [%s] read error\n", path);
    return NULL;
  }

  imlib_context_set_image(imlib_image);

  /* After imlib_context_set_image() ? */
  *width = imlib_image_get_width();
  *height = imlib_image_get_height();

  log_width = (int)log2f((float)(*width));
  *tex_width = (int)powf(2, log_width);
  if(*width != *tex_width){
    log_width = log_width + 1;
    *tex_width = (int)powf(2, log_width);
  }

  log_height = (int)log2f((float)(*height));
  *tex_height = (int)powf(2, log_height);
  if(*height != *tex_height){
    log_height = log_height + 1;
    *tex_height = (int)powf(2, log_height);
  }

  image_data = imlib_image_get_data();

  pixels = malloc((*tex_width) * (*tex_height) * 3);
  if(pixels == NULL){
    perror("readImage(): malloc");
    imlib_free_image();
    return NULL;
  }

  i = 0;
  for(y = 0 ; y < *height ; y++){
    for(x = 0 ; x < *tex_width ; x++){
      if(x < *width){
        /* BGRA to GL_RGB */
        itoc.data_int = image_data[i];
        bv = itoc.data_char[0];
        gv = itoc.data_char[1];
        rv = itoc.data_char[2];
        i++;
      }else{
        bv = 0;
        gv = 0;
        rv = 0;
      }
      /* R */
      pixels[(y * (*tex_width) + x) * 3] = rv;

      /* G */
      pixels[(y * (*tex_width) + x) * 3 + 1] = gv;

      /* B */
      pixels[(y * (*tex_width) + x) * 3 + 2] = bv;
    }
  }

  for(y = *height ; y < *tex_height ; y++){
    for(x = 0 ; x < *tex_width ; x++){
      /* R */
      pixels[(y * (*tex_width) + x) * 3] = 0;

      /* G */
      pixels[(y * (*tex_width) + x) * 3 + 1] = 0;

      /* B */
      pixels[(y * (*tex_width) + x) * 3 + 2] = 0;
    }
  }

  imlib_free_image();
#endif /* USE_IMLIB2 */

   return pixels;
}

void redrawWindow(void)
{
  /* Window Clear */
  glClear(GL_COLOR_BUFFER_BIT);

  /* Set Viewport */
  glViewport(0, 0, winWidth, winHeight);

#ifdef DEBUG
printf("redrawWindow: Window: (%d, %d)\n", winWidth, winHeight);
#endif /* DEBUG */

  /* Back Ground Texture Draw */
  GLfloat bg_vertices[] = { -1.0f,  1.0f, 0.0f,  // Position 0
                            -1.0f, -1.0f, 0.0f,  // Position 1
                             1.0f, -1.0f, 0.0f,  // Position 2
                             1.0f,  1.0f, 0.0f   // Position 3
                          };

  GLushort indices[] = { 0, 1, 2, 0, 2, 3 };

  /* Use Back Ground Program */
  glUseProgram(bgProgram);

  /* Set Vertex Position */
  glVertexAttribPointer(bgPositionLoc, 3, GL_FLOAT,
                        GL_FALSE, 0, bg_vertices);
  glEnableVertexAttribArray(bgPositionLoc);

  /* Set Texture Coordinate */
  glVertexAttribPointer(bgTexCoordLoc, 2, GL_FLOAT,
                        GL_FALSE, 0, bgTexVertices);
  glEnableVertexAttribArray(bgTexCoordLoc);

  /* Bind Texture */
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, bgTexture);

  /* Set Sampler Texture Unit */
  glUniform1i(bgTextureLoc, 0);

  /* Draw */
  glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_SHORT, indices);

  /* Window View */
  eglSwapBuffers(eglDisplay, eglSurface);

  return;
}

void eventLoop(void)
{
  XEvent event;
  KeySym key_sym;
  int e_loop = 1;
  int ret_key;
  char ret_string[2];

  while(e_loop){
    XNextEvent(xDisplay, &event);
    switch(event.type){
    case ClientMessage:
      if((Atom)event.xclient.data.l[0] == wmDeleteWindow){
        e_loop = 0;
      }
      break;
    case Expose:
      redrawWindow();
      break;
    case MappingNotify:
      XRefreshKeyboardMapping(&(event.xmapping));
      break;
    case KeyPress:
      ret_string[0] = ret_string[1] = 0x00;

      ret_key = XLookupString(&(event.xkey), ret_string, 1, &key_sym, NULL);
      if(ret_key < 0){
        continue;
      }

      switch(key_sym){
      case XK_Q:
      case XK_q:
        e_loop = 0;
        break;
      }
      break;
    }
  }

  return;
}

void closeWindow(void)
{
  if(eglDisplay != EGL_NO_DISPLAY){
    eglMakeCurrent(eglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);

    if(bgTexture != 0){
      glDeleteTextures(1, &bgTexture);
      bgTexture = 0;
    }

    if(bgProgram != 0){
      glDeleteProgram(bgProgram);
      bgProgram = 0;
    }

    if(eglContext != EGL_NO_CONTEXT){
       eglDestroyContext(eglDisplay, eglContext);
       eglContext = EGL_NO_CONTEXT;
    }

    if(eglSurface != EGL_NO_SURFACE){
       eglDestroySurface(eglDisplay, eglSurface);
       eglSurface = EGL_NO_SURFACE;
    }

    eglTerminate(eglDisplay);
    eglDisplay = EGL_NO_DISPLAY;
  }

  if(xDisplay != NULL){
    if(xWindow != 0){
      XDestroyWindow(xDisplay, xWindow);
      xWindow = 0;
    }

    XCloseDisplay(xDisplay);
    xDisplay = NULL;
  }

  return;
}

GLuint loadShader(const char *shader_source, GLenum type)
{
  GLuint shader;
  GLint success;
  GLint info_len = 0;
  char* info_log;

  /* Create Shader Object */
  shader = glCreateShader(type);
  if(shader == 0){
    return 0;
  }

  /* Load Shader Source */
  glShaderSource(shader, 1, &shader_source, NULL);

  /* Compile Shader */
  glCompileShader(shader);

  /* Check Compile */
  glGetShaderiv(shader, GL_COMPILE_STATUS, &success);
  if(success != GL_TRUE){
    /* Get Error Message */
    glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &info_len);
    if(info_len > 1){
      info_log = malloc(sizeof(char) * info_len);
      if(info_log != NULL){
        glGetShaderInfoLog(shader, info_len, NULL, info_log);
        if(type == GL_VERTEX_SHADER){
          printf("Compile Vertex Shader::\n%s\n", info_log);
        }else
        if(type == GL_FRAGMENT_SHADER){
          printf("Compile Fragment Shader::\n%s\n", info_log);
        }else{
          printf("Compile Shader::\n%s\n", info_log);
        }
        free(info_log);
      }
    }

    glDeleteShader(shader);
    return 0;
  }

  return shader;
}

GLuint loadProgram(const char *vertex_src, const char *fragment_src)
{
  GLuint vertex_shader;
  GLuint fragment_shader;
  GLuint program_object;
  GLint success;
  GLint info_len = 0;
  char* info_log;

  /* Load Vertex and Fragment shaders */
  vertex_shader = loadShader(vertex_src, GL_VERTEX_SHADER);
  if(vertex_shader == 0){
    return 0;
  }

  fragment_shader = loadShader(fragment_src, GL_FRAGMENT_SHADER);
  if(fragment_shader == 0){
    glDeleteShader(vertex_shader);
    return 0;
  }

  /* Create Program Object */
  program_object = glCreateProgram();
  if(program_object == 0){
    glDeleteShader(vertex_shader);
    glDeleteShader(fragment_shader);
    return 0;
  }

  glAttachShader(program_object, vertex_shader);
  glAttachShader(program_object, fragment_shader);

  /* Link Program */
  glLinkProgram(program_object);

  /* Check Link */
  glGetProgramiv(program_object, GL_LINK_STATUS, &success);
  if(success != GL_TRUE){
    /* Get Error Message */
    glGetProgramiv(program_object, GL_INFO_LOG_LENGTH, &info_len);
    if(info_len > 1){
      info_log = malloc(sizeof(char) * info_len);
      if(info_log != NULL){
        glGetProgramInfoLog(program_object, info_len, NULL, info_log);
        printf("Link Shader Program::\n%s\n", info_log);
        free(info_log);
      }
    }

    glDeleteProgram(program_object);
    return 0;
  }

  /* Free Shader Object */
  glDeleteShader(vertex_shader);
  glDeleteShader(fragment_shader);

  return program_object;
}

void usageHelp(char *cmd)
{
  printf("%s %s Copyright (c) 2012 Sano Yukihiko\n",
         cmd, egldisplayVersion);

  printf("usage: %s", cmd);
  printf(" file");
  printf("\n");

  return;
}

int main(int ac, char **av)
{
  int i;
  int img_width;
  int img_height;
  int tex_width;
  int tex_height;
  char *file_name = NULL;
  GLubyte *pixels;
  GLfloat wf;
  EGLConfig egl_config;
  EGLint num_config;
  XSizeHints xsh;
  XTextProperty w_title_property;
  XTextProperty i_title_property;

  EGLint attribute_list[] = {
#ifdef PANDORA
    EGL_LEVEL, 0,
    EGL_NATIVE_RENDERABLE, EGL_FALSE,
    EGL_DEPTH_SIZE, EGL_DONT_CARE,
#endif /* PANDORA */
    EGL_RED_SIZE, 0,
    EGL_GREEN_SIZE, 0,
    EGL_BLUE_SIZE, 0,
    EGL_BUFFER_SIZE, 0,
    EGL_SURFACE_TYPE, EGL_WINDOW_BIT,
    EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT,
    EGL_NONE
  };

  EGLint context_attribute_list[] = {
    EGL_CONTEXT_CLIENT_VERSION, 2,
    EGL_NONE
  };

  cmdName = NULL;

  /* Basename Command */
  cmdName = strrchr(av[0], '/');
  if(cmdName == NULL){
    cmdName = av[0];
  }else{
    cmdName++;
  }

  for(i = 1 ; i < ac ; i++){
    if((strcmp(av[i], "-h")) == 0
    || (strcmp(av[i], "--help")) == 0){
      usageHelp(cmdName);
      exit(1);
    }
    else{
      if(av[i][0] == '-'){
        usageHelp(cmdName);
        exit(1);
      }
      if(file_name != NULL){
        continue;
      }
      file_name = av[i];
    }
  }

  if(file_name == NULL){
    usageHelp(cmdName);
    exit(1);
  }

  xDisplay = XOpenDisplay(NULL);
  if(xDisplay == NULL){
    printf("XOpenDisplay: error\n");
    exit(1);
  }

  /* Image File Read */
  /* Read Back Ground Texture */
  pixels = readImage(file_name, &img_width, &img_height,
                                &tex_width, &tex_height);
  if(pixels == NULL){
    closeWindow();
    exit(1);
  }

  winWidth = img_width;
  winHeight = img_height;

  if(tex_width > img_width){
    wf = (GLfloat)img_width / tex_width;
    bgTexVertices[4] = bgTexVertices[6] = wf;
  }

  if(tex_height > img_height){
    wf = (GLfloat)img_height / tex_height;
    bgTexVertices[3] = bgTexVertices[5] = wf;
  }

#ifdef DEBUG
printf("Window  : (%d, %d)\n", winWidth, winHeight);
printf("Image   : (%d, %d)\n", img_width, img_height);
printf("Texture : (%d, %d)\n", tex_width, tex_height);
printf("(%f, %f)\n(%f, %f)\n(%f, %f)\n(%f, %f)\n",
       bgTexVertices[0],
       bgTexVertices[1],
       bgTexVertices[2],
       bgTexVertices[3],
       bgTexVertices[4],
       bgTexVertices[5],
       bgTexVertices[6],
       bgTexVertices[7]);
#endif /* DEBUG */

  xWindow = XCreateSimpleWindow(xDisplay,
                                DefaultRootWindow(xDisplay),
                                0, 0,
                                winWidth, winHeight,
                                1,
                                BlackPixel(xDisplay, DefaultScreen(xDisplay)),
                                WhitePixel(xDisplay, DefaultScreen(xDisplay)));

  xsh.flags = (USSize | PMinSize | PMaxSize);
  xsh.width = winWidth;
  xsh.height = winHeight;
  xsh.min_width = winWidth;
  xsh.min_height = winHeight;
  xsh.max_width = winWidth;
  xsh.max_height = winHeight;

  XSetWMNormalHints(xDisplay, xWindow, &xsh);

  if(XmbTextListToTextProperty(xDisplay,
                               (char **)&cmdName,
                               1,
                               XStdICCTextStyle,
                               &w_title_property) >= Success){
    XSetWMName(xDisplay, xWindow, &w_title_property);
    XFree(w_title_property.value);
  }else{
    XStoreName(xDisplay, xWindow, cmdName);
  }

  if(XmbTextListToTextProperty(xDisplay,
                               (char **)&cmdName,
                               1,
                               XStdICCTextStyle,
                               &i_title_property) >= Success){
    XSetWMIconName(xDisplay, xWindow, &i_title_property);
    XFree(i_title_property.value);
  }else{
    XSetIconName(xDisplay, xWindow, cmdName);
  }

  /* Set Check Event */
  XSelectInput(xDisplay, xWindow, ExposureMask | KeyPressMask);

  wmDeleteWindow = XInternAtom(xDisplay, "WM_DELETE_WINDOW", False);
  XSetWMProtocols(xDisplay, xWindow, &wmDeleteWindow, 1);

  /* OpenGL ES */
  eglDisplay = eglGetDisplay((EGLNativeDisplayType)xDisplay);
  if(eglDisplay == EGL_NO_DISPLAY){
    printf("eglGetDisplay: error\n");
    closeWindow();
    exit(1);
  }

  if(eglInitialize(eglDisplay, NULL, NULL) == EGL_FALSE){
    printf("eglInitialize: error\n");
    closeWindow();
    exit(1);
  }

  eglBindAPI(EGL_OPENGL_ES_API);

  switch(DefaultDepth(xDisplay, DefaultScreen(xDisplay))){
  case 32:
  case 24:
    for(i = 0 ; attribute_list[i] != EGL_NONE ; i+=2){
      switch(attribute_list[i]){
      case EGL_RED_SIZE:
      case EGL_GREEN_SIZE:
      case EGL_BLUE_SIZE:
        attribute_list[i + 1] = 8;
        break;
      case EGL_BUFFER_SIZE:
        attribute_list[i + 1] = DefaultDepth(xDisplay, DefaultScreen(xDisplay));
        break;
      }
    }
    break;
  case 16:
    for(i = 0 ; attribute_list[i] != EGL_NONE ; i+=2){
      switch(attribute_list[i]){
      case EGL_RED_SIZE:
      case EGL_BLUE_SIZE:
        attribute_list[i + 1] = 5;
        break;
      case EGL_GREEN_SIZE:
        attribute_list[i + 1] = 6;
        break;
      case EGL_BUFFER_SIZE:
        attribute_list[i + 1] = DefaultDepth(xDisplay, DefaultScreen(xDisplay));
        break;
      }
    }
    break;
  default:
    printf("not support color depth (%d)\n",
           DefaultDepth(xDisplay, DefaultScreen(xDisplay)));
    closeWindow();
    exit(1);
    break;
  }

  num_config = 0;
  eglChooseConfig(eglDisplay, attribute_list, &egl_config, 1, &num_config);
  if(num_config == 0){
    printf("eglChooseConfig: error\n");
    closeWindow();
    exit(1);
  }

  eglContext = eglCreateContext(eglDisplay, egl_config, EGL_NO_CONTEXT,
                                context_attribute_list);
  if(eglContext == EGL_NO_CONTEXT){
    printf("eglCreateContext: error\n");
    closeWindow();
    exit(1);
  }

  eglSurface = eglCreateWindowSurface(eglDisplay, egl_config,
                                      (EGLNativeWindowType)xWindow, NULL);
  if(eglSurface == EGL_NO_SURFACE){
    printf("eglCreateWindowSurface: error\n");
    closeWindow();
    exit(1);
  }

  if(eglMakeCurrent(eglDisplay,
                    eglSurface, eglSurface, eglContext) != EGL_TRUE){
    printf("eglMakeCurrent: error\n");
    closeWindow();
    exit(1);
  }

  /* Back Ground Program */
  const char bg_vertex_src[] =
    "attribute vec4 a_position;    \n"
    "attribute vec2 a_texCoord;    \n"
    "varying vec2 v_texCoord;      \n"
    "                              \n"
    "void main()                   \n"
    "{                             \n"
    "   gl_Position = a_position;  \n"
    "   v_texCoord = a_texCoord;   \n"
    "}                             \n";

  const char bg_fragment_src[] =
    "precision mediump float;                           \n"
    "varying vec2 v_texCoord;                           \n"
    "uniform sampler2D s_texture;                       \n"
    "                                                   \n"
    "void main()                                        \n"
    "{                                                  \n"
    "  gl_FragColor = texture2D(s_texture, v_texCoord); \n"
    "}                                                  \n";

  bgProgram = loadProgram(bg_vertex_src, bg_fragment_src);
  if(bgProgram == 0){
    closeWindow();
    exit(1);
  }

  /* Get Attribute Locations */
  bgPositionLoc = glGetAttribLocation(bgProgram, "a_position");
  bgTexCoordLoc = glGetAttribLocation(bgProgram, "a_texCoord");
  if(bgPositionLoc < 0 || bgTexCoordLoc < 0){
    printf("glGetAttribLocation: error\n");
    closeWindow();
    exit(1);
  }

  /* Get Sampler Locations */
  bgTextureLoc = glGetUniformLocation(bgProgram, "s_texture");
  if(bgTextureLoc < 0){
    printf("glGetUniformLocation error\n");
    closeWindow();
    exit(1);
  }

  /* Back Ground Texture */
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

  /* Generate Texture Object */
  glGenTextures(1, &bgTexture);

  /* Bind Texture Object */
  glBindTexture(GL_TEXTURE_2D, bgTexture);

  /* Load Texture */
  glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB,
               tex_width, tex_height, 0, GL_RGB, GL_UNSIGNED_BYTE, pixels);

  /* Set Filtering Mode */
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

  free(pixels);

  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

  /* Map Window */
  XMapWindow(xDisplay, xWindow);

  XSync(xDisplay, False);

  /* Event Loop */
  eventLoop();

  closeWindow();

  exit(0);
}
