#define EGLNUBSAMPLE_VERSION "version 0.1"
char *eglNubSampleVersion = EGLNUBSAMPLE_VERSION;

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <EGL/egl.h>
#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#ifdef PANDORA
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/fb.h>

#ifndef FBIO_WAITFORVSYNC
#define FBIO_WAITFORVSYNC _IOW('F', 0x20, __u32)
#endif /* FBIO_WAITFORVSYNC */
#endif /* PANDORA */

#ifdef USE_IMLIB2
#include <Imlib2.h>
#endif /* USE_IMLIB2 */

#include "nub.h"

#define NUB_SPEED 48
#define NUB_SPEED_MIN 1
#define NUB_SPEED_MAX 256

#define WINDOW_WIDTH 800
#define WINDOW_HEIGHT 480

#define FPS_LIMIT 60

#define BG_TEXTURE_FILE "bg_512x512.png"
#define FPS_TEXTURE_FILE "fps_str.png"

#define FPS_LIMIT_MIN 15
#define FPS_LIMIT_MAX 180

/* X Window System */
void closeWindow(void);
void eventLoop(void);

/* OpenGL ES */
void drawBackGround(void);
void drawFpsString(void);
void drawObject1(void);
void drawTexture(GLfloat *dst, int d, GLfloat *src, int s);
void drawTriangle(void);
GLuint loadProgram(const char *vertex_src, const char *fragment_src);
GLuint loadShader(const char *shader_source, GLenum type);
GLubyte *readTextureImage(char *path, int *width, int *height,
                          GLenum format, GLenum type);
void redrawWindow(void);

/* Other */
void usageHelp(char *cmd);
void uSleep(int usec);

#ifdef PANDORA
/* workaround ? */
float log2f(float x);
#endif /* PANDORA */

/* X Window System */
Display *xDisplay = NULL;
Window xWindow = 0;
Atom wmDeleteWindow;

/* OpenGL ES */
EGLDisplay eglDisplay = EGL_NO_DISPLAY;
EGLContext eglContext = EGL_NO_CONTEXT;
EGLSurface eglSurface = EGL_NO_SURFACE;

/* Object-1s */
GLuint object1Program = 0;
GLboolean updatePos = GL_FALSE;
GLint phaseLoc;
GLint offsetLoc;
GLint colorLoc;
GLint positionLoc;
GLfloat vertexArray[] = {
   0.0,  0.3,  0.0,
  -0.3,  0.0,  0.0,
   0.0, -0.3,  0.0,
   0.3,  0.0,  0.0,
   0.0,  0.3,  0.0
};

/* Object-1 */
GLfloat newX[2] = { -0.5,  0.5 };
GLfloat newY[2] = {  0.0,  0.0 };
GLfloat offsetX[2] = { -0.5,  0.5 };
GLfloat offsetY[2] = {  0.0,  0.0 };
GLfloat oldX[2] = { -0.5,  0.5 };
GLfloat oldY[2] = {  0.0,  0.0 };
GLfloat objectColor[2][4] = {{0.0, 0.0, 1.0, 1.0},
                             {1.0, 0.0, 0.0, 1.0}
                            };

/* Back Ground Texture */
GLuint bgProgram = 0;
GLuint bgTexture = 0;
GLint bgPositionLoc;
GLint bgTexCoordLoc;
GLint bgTextureLoc;

GLfloat bgTexVertices[] = {  0.0,  0.0,  // TexCoord 0
                             0.0,  1.0,  // TexCoord 1
                             1.0,  1.0,  // TexCoord 2
                             1.0,  0.0   // TexCoord 3
                          };

/* Fps Char Texture */
GLuint fpsTexture = 0;

/* Triangle */
GLuint triangleProgram = 0;
GLint triangleLoc;
int triangleLeft = 0;
int triangleRight = 0;
int triangleUp = 0;
int triangleDown = 0;
GLfloat triangleVertices[] = { -0.88,  0.33, 0.0,
                               -0.97,  0.0,  0.0,
                               -0.88, -0.33, 0.0
                             };

/* Command Option */
int nubSpeed = NUB_SPEED;
int fpsLimit = FPS_LIMIT;
int fpsNum = 0;
int fpsPrint = 1;
char *cmdName;
#ifdef PANDORA
int fbDev = -1;
#endif /* PANDORA */

void drawBackGround(void)
{
  GLfloat bg_vertices[] = { -1.0,  1.0, 0.0,  // Position 0
                            -1.0, -1.0, 0.0,  // Position 1
                             1.0, -1.0, 0.0,  // Position 2
                             1.0,  1.0, 0.0   // Position 3
                          };

  /* Use Back Ground Program */
  glUseProgram(bgProgram);

  /* Bind Texture */
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, bgTexture);

  /* Set Sampler Texture Unit */
  glUniform1i(bgTextureLoc, 0);

  drawTexture(bg_vertices, 3, bgTexVertices, 2);

  return;
}

void drawFpsString(void)
{
  int i;
  int x;
  int y;
  int fps_1;
  int fps_10;
  int fps_100;
  GLfloat fps_dst_vertices[] = { -0.95,  0.9, 0.0, // Position 0
                                 -0.95,  0.7, 0.0, // Position 1
                                 -0.7,   0.7, 0.0, // Position 2
                                 -0.7,   0.9, 0.0  // Position 3
                               };
  GLfloat fps_src_vertices[] = {  0.0,  0.75, // TexCoord 0
                                  0.0,  1.0,  // TexCoord 1
                                  1.0,  1.0,  // TexCoord 2
                                  1.0,  0.75  // TexCoord 3
                               };

  /* Use Back Ground Program */
  glUseProgram(bgProgram);

  /* Bind Texture */
  glActiveTexture(GL_TEXTURE0);
  glBindTexture(GL_TEXTURE_2D, fpsTexture);

  /* Set Sampler Texture Unit */
  glUniform1i(bgTextureLoc, 0);

  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);

  /* "fps : " */
  //fps_src_vertices[0] = fps_src_vertices[2] = 0.0;
  //fps_src_vertices[1] = fps_src_vertices[7] = 0.75;
  //fps_src_vertices[3] = fps_src_vertices[5] = 1.0;
  //fps_src_vertices[4] = fps_src_vertices[6] = 1.0;

  drawTexture(fps_dst_vertices, 3, fps_src_vertices, 2);

  /* 100 */
  fps_100 = fpsNum / 100;
  i = fpsNum - fps_100 * 100;
  fps_10 = i / 10;
  fps_1 = i - fps_10 * 10;
  i = 0;

  if(fps_100 != 0){
    fps_dst_vertices[0] = fps_dst_vertices[3] = -0.71;
    fps_dst_vertices[6] = fps_dst_vertices[9] = -0.635;

    x = fps_100 % 4;
    y = fps_100 / 4;

    fps_src_vertices[0] = fps_src_vertices[2] = (GLfloat)1 * x / 4;
    fps_src_vertices[1] = fps_src_vertices[7] = (GLfloat)1 * y / 4;
    fps_src_vertices[3] = fps_src_vertices[5] = fps_src_vertices[1] + 0.25;
    fps_src_vertices[4] = fps_src_vertices[6] = fps_src_vertices[0] + 0.25;

    drawTexture(fps_dst_vertices, 3, fps_src_vertices, 2);

    i = 1;
  }

  /* 10 */
  if(fps_10 != 0 || i == 1){
    fps_dst_vertices[0] = fps_dst_vertices[3] = -0.635;
    fps_dst_vertices[6] = fps_dst_vertices[9] = -0.56;

    x = fps_10 % 4;
    y = fps_10 / 4;

    fps_src_vertices[0] = fps_src_vertices[2] = (GLfloat)1 * x / 4;
    fps_src_vertices[1] = fps_src_vertices[7] = (GLfloat)1 * y / 4;
    fps_src_vertices[3] = fps_src_vertices[5] = fps_src_vertices[1] + 0.25;
    fps_src_vertices[4] = fps_src_vertices[6] = fps_src_vertices[0] + 0.25;

    drawTexture(fps_dst_vertices, 3, fps_src_vertices, 2);
  }

  /* 1 */
  fps_dst_vertices[0] = fps_dst_vertices[3] = -0.56;
  fps_dst_vertices[6] = fps_dst_vertices[9] = -0.485;

  x = fps_1 % 4;
  y = fps_1 / 4;

  fps_src_vertices[0] = fps_src_vertices[2] = (GLfloat)1 * x / 4;
  fps_src_vertices[1] = fps_src_vertices[7] = (GLfloat)1 * y / 4;
  fps_src_vertices[3] = fps_src_vertices[5] = fps_src_vertices[1] + 0.25;
  fps_src_vertices[4] = fps_src_vertices[6] = fps_src_vertices[0] + 0.25;

  drawTexture(fps_dst_vertices, 3, fps_src_vertices, 2);

  glDisable(GL_BLEND);

  return;
}

void drawTexture(GLfloat *dst, int d, GLfloat *src, int s)
{
  GLushort indices[] = { 0, 1, 2, 0, 2, 3 };

  /* Set Vertex Position */
  glVertexAttribPointer(bgPositionLoc, d, GL_FLOAT, GL_FALSE, 0, dst);
  glEnableVertexAttribArray(bgPositionLoc);

  /* Set Texture Coordinate */
  glVertexAttribPointer(bgTexCoordLoc, s, GL_FLOAT, GL_FALSE, 0, src);
  glEnableVertexAttribArray(bgTexCoordLoc);

  /* Draw */
  glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_SHORT, indices);

  return;
}

void drawTriangle(void)
{
  if(triangleLeft == 1){
    triangleVertices[0] = -0.88;
    triangleVertices[1] = 0.33;

    triangleVertices[3] = -0.97;
    triangleVertices[4] = 0.0;

    triangleVertices[6] = -0.88;
    triangleVertices[7] = -0.33;
  }

  if(triangleRight == 1){
    triangleVertices[0] = 0.88;
    triangleVertices[1] = 0.33;

    triangleVertices[3] = 0.97;
    triangleVertices[4] = 0.0;

    triangleVertices[6] = 0.88;
    triangleVertices[7] = -0.33;
  }

  if(triangleUp == 1){
    triangleVertices[0] = 0.0;
    triangleVertices[1] = 0.95;

    triangleVertices[3] = -0.2;
    triangleVertices[4] = 0.8;

    triangleVertices[6] = 0.2;
    triangleVertices[7] = 0.8;
  }

  if(triangleDown == 1){
    triangleVertices[0] = 0.0;
    triangleVertices[1] = -0.95;

    triangleVertices[3] = -0.2;
    triangleVertices[4] = -0.8;

    triangleVertices[6] = 0.2;
    triangleVertices[7] = -0.8;
  }

  /* Use Triangle Program */
  glUseProgram(triangleProgram);

  glVertexAttribPointer(triangleLoc, 3, GL_FLOAT, GL_FALSE, 0,
                        triangleVertices);
  glEnableVertexAttribArray(triangleLoc);

  glDrawArrays(GL_TRIANGLES, 0, 3);

  return;
}

void drawObject1(void)
{
  int i;
  static float phase = 0;
  GLfloat old_offset_x;
  GLfloat old_offset_y;

  /* Use Object-1 Program */
  glUseProgram(object1Program);

  glUniform1f(phaseLoc, phase);
  phase = fmodf(phase + 0.5, 2.0 * 3.141);

  if(updatePos == GL_TRUE){
    /* Object-1 */
    for(i = 0 ; i < 2 ; i++){
      old_offset_x = offsetX[i];
      old_offset_y = offsetY[i];

      offsetX[i] = newX[i] - oldX[i];
      offsetY[i] = newY[i] - oldY[i];

      oldX[i] = newX[i];
      oldY[i] = newY[i];

      offsetX[i] += old_offset_x;
      offsetY[i] += old_offset_y;
    }

    updatePos = GL_FALSE;
  }

  glVertexAttribPointer(positionLoc, 3, GL_FLOAT, GL_FALSE, 0, vertexArray);
  glEnableVertexAttribArray(positionLoc);

  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);

  for(i = 0 ; i < 2 ; i++){
    glUniform4f(offsetLoc, offsetX[i], offsetY[i], 0.0, 0.0);
    glUniform4f(colorLoc,
                objectColor[i][0], objectColor[i][1],
                objectColor[i][2], objectColor[i][3]);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 5);
  }

  glDisable(GL_BLEND);

  return;
}

void redrawWindow(void)
{
  /* Window Clear */
  glClear(GL_COLOR_BUFFER_BIT);

  /* Set Viewport */
  glViewport(0, 0, WINDOW_WIDTH, WINDOW_HEIGHT);

  /* Back Ground Texture Draw */
  drawBackGround();

  /* Object-1 Draw */
  drawObject1();

  if(triangleLeft == 1 || triangleRight == 1
  || triangleUp == 1 || triangleDown == 1){
    drawTriangle();
  }

  /* Fps Char Texture Draw */
  if(fpsPrint == 1){
    drawFpsString();
  }

#ifdef PANDORA
  if(fbDev >= 0){
    int ret;
    int fb_vsync = 0;

    ret = ioctl(fbDev, FBIO_WAITFORVSYNC, &fb_vsync);
    if(ret == -1){
      perror("ioctl");
    }
  }
#endif /* PANDORA */

  /* Window View */
  eglSwapBuffers(eglDisplay, eglSurface);

  return;
}

void eventLoop(void)
{
  XEvent event;
  KeySym key_sym;
  int i;
  int e_loop = 1;
  int ret_key;
  int num_frames = 0;
  int usecframe;
  int setfps;
  int timeo;
  float num_fps;
  char ret_string[2];
  struct timezone tz;
  struct timeval startfps, endfps;
  struct timeval startframe, endframe;
  GLfloat val;
  GLfloat absm;

  gettimeofday(&startfps, &tz);

  while(e_loop){
    gettimeofday(&startframe, &tz);

    if(XPending(xDisplay)){
      XNextEvent(xDisplay, &event);
      switch(event.type){
      case ClientMessage:
        if((Atom)event.xclient.data.l[0] == wmDeleteWindow){
          e_loop = 0;
        }
        break;
      case MappingNotify:
        XRefreshKeyboardMapping(&(event.xmapping));
        break;
      case KeyPress:
      case KeyRelease:
        ret_string[0] = ret_string[1] = 0x00;

        ret_key = XLookupString(&(event.xkey),
                                ret_string, 1, &key_sym, NULL);
        if(ret_key < 0){
          continue;
        }

        switch(key_sym){
        case XK_Home:      /* Pandora Button A */
          if(event.type == KeyRelease){
            if(vertexArray[1] != 0.15){
              vertexArray[1] = vertexArray[9] = vertexArray[13] = 0.15;
              vertexArray[3] = vertexArray[7] = -0.15;
            }
          }
          break;
        case XK_End:       /* Pandora Button B */
          if(event.type == KeyRelease){
            if(vertexArray[1] != 0.45){
              vertexArray[1] = vertexArray[9] = vertexArray[13] = 0.45;
              vertexArray[3] = vertexArray[7] = -0.45;
            }
          }
          break;
        case XK_Page_Up:   /* Pandora Button Y */
          if(event.type == KeyRelease){
            if(vertexArray[1] != 0.3){
              vertexArray[1] = vertexArray[9] = vertexArray[13] = 0.3;
              vertexArray[3] = vertexArray[7] = -0.3;
            }
          }
          break;
        case XK_Page_Down: /* Pandora Button X */
          if(event.type == KeyRelease){
            e_loop = 0;
          }
          break;
        case XK_Alt_L:     /* Pandora Button START */
          if(event.type == KeyRelease){
            fpsPrint = 1;
          }
          break;
        case XK_Control_L: /* Pandora Button SELECT */
          if(event.type == KeyRelease){
            fpsPrint = 0;
          }
          break;
        //case XK_XXXX:      /* Pandora Button PANDORA (MENU 1008ff65 ?) */
        //  break;
        case XK_Shift_R:   /* Pandora Shoulder L */
          if(event.type == KeyPress){
            objectColor[0][0] = 0.0;
            objectColor[0][1] = 1.0;
            objectColor[0][2] = 1.0;
            objectColor[0][3] = 1.0;
          }else{
            objectColor[0][0] = 0.0;
            objectColor[0][1] = 0.0;
            objectColor[0][2] = 1.0;
            objectColor[0][3] = 1.0;
          }
          break;
        case XK_Control_R: /* Pandora Shoulder R */
          if(event.type == KeyPress){
            objectColor[1][0] = 1.0;
            objectColor[1][1] = 1.0;
            objectColor[1][2] = 0.0;
            objectColor[1][3] = 1.0;
          }else{
            objectColor[1][0] = 1.0;
            objectColor[1][1] = 0.0;
            objectColor[1][2] = 0.0;
            objectColor[1][3] = 1.0;
          }
          break;
        case XK_Left:      /* Pandora D-Pad L */
          if(event.type == KeyPress){
            triangleLeft = 1;
          }else{
            triangleLeft = 0;
          }
          break;
        case XK_Right:     /* Pandora D-Pad R */
          if(event.type == KeyPress){
            triangleRight = 1;
          }else{
            triangleRight = 0;
          }
          break;
        case XK_Up:        /* Pandora D-Pad U */
          if(event.type == KeyPress){
            triangleUp = 1;
          }else{
            triangleUp = 0;
          }
          break;
        case XK_Down:      /* Pandora D-Pad D */
          if(event.type == KeyPress){
            triangleDown = 1;
          }else{
            triangleDown = 0;
          }
          break;
        case XK_q:
          if(event.type == KeyRelease){
            e_loop = 0;
          }
          break;
        }

        break;
      }
    }

    nubRead();

    for(i = 0 ; i < nubNums ; i++){
      val = nubInfo[i].value_x;
      absm = nubInfo[i].abs_x_mult / nubSpeed;
      newX[i] = newX[i] + (val / absm);
      if(newX[i] > 1.0){
        newX[i] = 1.0;
      }
      if(newX[i] < -1.0){
        newX[i] = -1.0;
      }

      val = nubInfo[i].value_y;
      absm = nubInfo[i].abs_y_mult / nubSpeed;
      newY[i] = newY[i] - (val / absm);
      if(newY[i] > 1.0){
        newY[i] = 1.0;
      }
      if(newY[i] < -1.0){
        newY[i] = -1.0;
      }

      updatePos = GL_TRUE;
    }

    redrawWindow();

    gettimeofday(&endframe, &tz);

    usecframe = (endframe.tv_sec - startframe.tv_sec) * 1000 +
                (endframe.tv_usec - startframe.tv_usec);
    if(usecframe < 0){
      usecframe = 0;
    }
    setfps = (fpsLimit / 60);
    if(setfps < 1){
      setfps = 1;
    }
    if(setfps > 3){
      setfps = 3;
    }
    setfps = fpsLimit + setfps;
    if(usecframe < (1000000 / setfps)){
      timeo = (1000000 / setfps) - usecframe;
      uSleep(timeo);
    }

    if(++num_frames % 100 == 0){
      gettimeofday(&endfps, &tz);
      num_fps = endfps.tv_sec - startfps.tv_sec +
                (endfps.tv_usec - startfps.tv_usec) * (1.0 / 1000000);
      fpsNum = (int)(num_frames / num_fps);
      //printf("fps: %d\n", fpsNum);
      num_frames = 0;
      startfps = endfps;
    }
  }

  return;
}

void closeWindow(void)
{
  if(eglDisplay != EGL_NO_DISPLAY){
    eglMakeCurrent(eglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
    glBindTexture(GL_TEXTURE_2D, 0);

    if(bgTexture != 0){
      glDeleteTextures(1, &bgTexture);
      bgTexture = 0;
    }

    if(fpsTexture != 0){
      glDeleteTextures(1, &fpsTexture);
      fpsTexture = 0;
    }

    if(bgProgram != 0){
      glDeleteProgram(bgProgram);
      bgProgram = 0;
    }

    if(object1Program != 0){
      glDeleteProgram(object1Program);
      object1Program = 0;
    }

    if(triangleProgram != 0){
      glDeleteProgram(triangleProgram);
      triangleProgram = 0;
    }

    if(eglContext != EGL_NO_CONTEXT){
       eglDestroyContext(eglDisplay, eglContext);
       eglContext = EGL_NO_CONTEXT;
    }

    if(eglSurface != EGL_NO_SURFACE){
       eglDestroySurface(eglDisplay, eglSurface);
       eglSurface = EGL_NO_SURFACE;
    }

    eglTerminate(eglDisplay);
    eglDisplay = EGL_NO_DISPLAY;
  }

  if(xDisplay != NULL){
    if(xWindow != 0){
      XDestroyWindow(xDisplay, xWindow);
      xWindow = 0;
    }

    XCloseDisplay(xDisplay);
    xDisplay = NULL;
  }

  nubClose();

#ifdef PANDORA
  if(fbDev >= 0){
    close(fbDev);
    fbDev = -1;
  }
#endif /* PANDORA */

  return;
}

GLuint loadShader(const char *shader_source, GLenum type)
{
  GLuint shader;
  GLint success;
  GLint info_len = 0;
  char* info_log;

  /* Create Shader Object */
  shader = glCreateShader(type);
  if(shader == 0){
    return 0;
  }

  /* Load Shader Source */
  glShaderSource(shader, 1, &shader_source, NULL);

  /* Compile Shader */
  glCompileShader(shader);

  /* Check Compile */
  glGetShaderiv(shader, GL_COMPILE_STATUS, &success);
  if(success != GL_TRUE){
    /* Get Error Message */
    glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &info_len);
    if(info_len > 1){
      info_log = malloc(sizeof(char) * info_len);
      if(info_log != NULL){
        glGetShaderInfoLog(shader, info_len, NULL, info_log);
        if(type == GL_VERTEX_SHADER){
          printf("Compile Vertex Shader::\n%s\n", info_log);
        }else
        if(type == GL_FRAGMENT_SHADER){
          printf("Compile Fragment Shader::\n%s\n", info_log);
        }else{
          printf("Compile Shader::\n%s\n", info_log);
        }
        free(info_log);
      }
    }

    glDeleteShader(shader);
    return 0;
  }

  return shader;
}

GLuint loadProgram(const char *vertex_src, const char *fragment_src)
{
  GLuint vertex_shader;
  GLuint fragment_shader;
  GLuint program_object;
  GLint success;
  GLint info_len = 0;
  char* info_log;

  /* Load Vertex and Fragment shaders */
  vertex_shader = loadShader(vertex_src, GL_VERTEX_SHADER);
  if(vertex_shader == 0){
    return 0;
  }

  fragment_shader = loadShader(fragment_src, GL_FRAGMENT_SHADER);
  if(fragment_shader == 0){
    glDeleteShader(vertex_shader);
    return 0;
  }

  /* Create Program Object */
  program_object = glCreateProgram();
  if(program_object == 0){
    glDeleteShader(vertex_shader);
    glDeleteShader(fragment_shader);
    return 0;
  }

  glAttachShader(program_object, vertex_shader);
  glAttachShader(program_object, fragment_shader);

  /* Link Program */
  glLinkProgram(program_object);

  /* Check Link */
  glGetProgramiv(program_object, GL_LINK_STATUS, &success);
  if(success != GL_TRUE){
    /* Get Error Message */
    glGetProgramiv(program_object, GL_INFO_LOG_LENGTH, &info_len);
    if(info_len > 1){
      info_log = malloc(sizeof(char) * info_len);
      if(info_log != NULL){
        glGetProgramInfoLog(program_object, info_len, NULL, info_log);
        printf("Link Shader Program::\n%s\n", info_log);
        free(info_log);
      }
    }

    glDeleteProgram(program_object);
    return 0;
  }

  /* Free Shader Object */
  glDeleteShader(vertex_shader);
  glDeleteShader(fragment_shader);

  return program_object;
}

GLubyte *readTextureImage(char *path, int *width, int *height,
                          GLenum format, GLenum type)
{
  GLubyte *pixels = NULL;

#ifdef USE_IMLIB2
  int i;
  int x;
  int y;
  int bitpr = 1;
  int bitpg = 1;
  int bitpb = 1;
  int bitpa = 1;
  int bitsr = 0;
  int bitsg = 0;
  int bitsb = 0;
  int bytepp;
  int log_width;
  int log_height;
  int new_width;
  int new_height;
  unsigned int ur = 0;
  unsigned int ug = 0;
  unsigned int ub = 0;
  unsigned int ua = 0;
  DATA32 *image_data;
  Imlib_Image imlib_image;

  union i_to_c {
    DATA32 data_int;
    GLubyte data_char[4];
  } itoc;

  union s_to_c {
    unsigned short data_short;
    GLubyte data_char[2];
  } stoc;

  if(type == GL_UNSIGNED_BYTE){
    if(format == GL_RGB){
      bytepp = 3;
    }else
    if(format == GL_RGBA){
      bytepp = 4;
    }else{
      return NULL;
    }
  }else
  if(type == GL_UNSIGNED_SHORT_5_6_5){
    if(format == GL_RGB){
      bytepp = 2;
      bitpr = 8;
      bitpg = 4;
      bitpb = 8;
      bitsr = 11;
      bitsg = 5;
      bitsb = 0;
      ua = 0;
    }else{
      return NULL;
    }
  }else
  if(type == GL_UNSIGNED_SHORT_5_5_5_1){
    if(format == GL_RGBA){
      bytepp = 2;
      bitpr = 8;
      bitpg = 8;
      bitpb = 8;
      bitpa = 255;
      bitsr = 11;
      bitsg = 6;
      bitsb = 1;
    }else{
      return NULL;
    }
  }else
  if(type == GL_UNSIGNED_SHORT_4_4_4_4){
    if(format == GL_RGBA){
      bytepp = 2;
      bitpr = 16;
      bitpg = 16;
      bitpb = 16;
      bitpa = 16;
      bitsr = 12;
      bitsg = 8;
      bitsb = 4;
    }else{
      return NULL;
    }
  }else{
    return NULL;
  }

  imlib_image = imlib_load_image(path);
  if(imlib_image == NULL){
    return NULL;
  }

  imlib_context_set_image(imlib_image);

  /* After imlib_context_set_image() ? */
  *width = imlib_image_get_width();
  *height = imlib_image_get_height();

  log_width = (int)log2f((float)(*width));
  new_width = (int)powf(2, log_width);
  if(*width != new_width){
    //printf("2^%d = %d\n", log_width, new_width);
    printf("Texture width (%d) is not n^2\n", *width);
    imlib_free_image();
    return NULL;
  }

  log_height = (int)log2f((float)(*height));
  new_height = (int)powf(2, log_height);
  if(*height != new_height){
    //printf("2^%d = %d\n", log_height, new_height);
    printf("Texture height (%d) is not n^2\n", *height);
    imlib_free_image();
    return NULL;
  }

  image_data = imlib_image_get_data();

  pixels = malloc((*width) * (*height) * bytepp);
  if(pixels == NULL){
    imlib_free_image();
    return NULL;
  }

  i = 0;
  for(y = 0 ; y < *height ; y++){
    for(x = 0 ; x < *width ; x++){
      /* BGRA to GL_RGBA */
      itoc.data_int = image_data[i];

      switch(type){
      case GL_UNSIGNED_BYTE:
        /* R */
        pixels[(y * (*width) + x) * bytepp] = itoc.data_char[2];

        /* G */
        pixels[(y * (*width) + x) * bytepp + 1] = itoc.data_char[1];

        /* B */
        pixels[(y * (*width) + x) * bytepp + 2] = itoc.data_char[0];

        if(format != GL_RGB){
          /* A */
          pixels[(y * (*width) + x) * bytepp + 3] = itoc.data_char[3];
        }

        break;
      case GL_UNSIGNED_SHORT_5_6_5:
      case GL_UNSIGNED_SHORT_5_5_5_1:
      case GL_UNSIGNED_SHORT_4_4_4_4:
        /* R */
        ur = (unsigned int)(itoc.data_char[2]) / bitpr;
        ur = ur<<bitsr;

        /* G */
        ug = (unsigned int)(itoc.data_char[1]) / bitpg;
        ug = ug<<bitsg;

        /* B */
        ub = (unsigned int)(itoc.data_char[0]) / bitpb;
        ub = ub<<bitsb;

        if(type == GL_UNSIGNED_SHORT_5_5_5_1 ||
           type == GL_UNSIGNED_SHORT_4_4_4_4){
          /* A */
          ua = (unsigned int)(itoc.data_char[3]) / bitpa;
        }

        stoc.data_short = ur | ug | ub | ua;
        pixels[(y * (*width) + x) * bytepp] = stoc.data_char[0];
        pixels[(y * (*width) + x) * bytepp + 1] = stoc.data_char[1];

        break;
      }

      i++;
    }
  }

  imlib_free_image();
#endif /* USE_IMLIB2 */

   return pixels;
}

void uSleep(int usec)
{
  int ret;
  struct timeval tval;

  tval.tv_sec  = 0;
  tval.tv_usec = usec;

  ret = select(0, NULL, NULL, NULL, &tval);
  if(ret == -1){
    perror("select");
  }

  return;
}

void usageHelp(char *cmd)
{
  printf("%s %s Copyright (c) 2012 Sano Yukihiko\n",
         cmd, eglNubSampleVersion);

  printf("usage: %s", cmd);
  printf(" [-fl fps]");
  printf(" [-ns speed]");
  printf("\n");

  printf("  -fl, --fps-limit\n");
  printf("    FPS limit setting (%d - %d) : default %d\n",
         FPS_LIMIT_MIN, FPS_LIMIT_MAX, FPS_LIMIT);

  printf("  -ns, --nub-speed\n");
  printf("    Nub speed setting (%d - %d) : default %d\n",
         NUB_SPEED_MIN, NUB_SPEED_MAX, NUB_SPEED);

  return;
}

int main(int ac, char **av)
{
  int i;
  int stat;
  int width;
  int height;
  GLubyte *pixels;
  GLenum format = GL_RGB;
  GLenum type = GL_UNSIGNED_BYTE;
  GLfloat wf;
  EGLConfig egl_config;
  EGLint num_config;
  XSizeHints xsh;
  XTextProperty w_title_property;
  XTextProperty i_title_property;
  XEvent xev;
  Atom nws_fullscreen;
  Atom nws;

  EGLint attribute_list[] = {
#ifdef PANDORA
    EGL_LEVEL, 0,
    EGL_NATIVE_RENDERABLE, EGL_FALSE,
    EGL_DEPTH_SIZE, EGL_DONT_CARE,
#endif /* PANDORA */
    EGL_RED_SIZE, 0,
    EGL_GREEN_SIZE, 0,
    EGL_BLUE_SIZE, 0,
    EGL_BUFFER_SIZE, 0,
    EGL_SURFACE_TYPE, EGL_WINDOW_BIT,
    EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT,
    EGL_NONE
  };

  EGLint context_attribute_list[] = {
    EGL_CONTEXT_CLIENT_VERSION, 2,
    EGL_NONE
  };

  cmdName = NULL;

  /* Basename Command */
  cmdName = strrchr(av[0], '/');
  if(cmdName == NULL){
    cmdName = av[0];
  }else{
    cmdName++;
  }

  for(i = 1 ; i < ac ; i++){
    if((strcmp(av[i], "-h")) == 0
    || (strcmp(av[i], "--help")) == 0){
      usageHelp(cmdName);
      exit(1);
    }
    else
    if((strcmp(av[i], "-ns")) == 0
    || (strcmp(av[i], "--nub-speed")) == 0){
      if((i + 1) >= ac){
        usageHelp(cmdName);
        exit(1);
      }
      stat = sscanf(av[++i], "%d", &nubSpeed);
      if(stat != 1){
        usageHelp(cmdName);
        exit(1);
      }
      if(nubSpeed < NUB_SPEED_MIN){
        nubSpeed = NUB_SPEED_MIN;
      }
      if(nubSpeed > NUB_SPEED_MAX){
        nubSpeed = NUB_SPEED_MAX;
      }
    }
    else
    if((strcmp(av[i], "-fl")) == 0
    || (strcmp(av[i], "--fps-limit")) == 0){
      if((i + 1) >= ac){
        usageHelp(cmdName);
        exit(1);
      }
      stat = sscanf(av[++i], "%d", &fpsLimit);
      if(stat != 1){
        usageHelp(cmdName);
        exit(1);
      }
      if(fpsLimit < FPS_LIMIT_MIN){
        fpsLimit = FPS_LIMIT_MIN;
      }
      if(fpsLimit > FPS_LIMIT_MAX){
        fpsLimit = FPS_LIMIT_MAX;
      }
    }
    else{
      usageHelp(cmdName);
      exit(1);
    }
  }

  xDisplay = XOpenDisplay(NULL);
  if(xDisplay == NULL){
    printf("XOpenDisplay: error\n");
    exit(1);
  }

  xWindow = XCreateSimpleWindow(xDisplay,
                                DefaultRootWindow(xDisplay),
                                0, 0,
                                WINDOW_WIDTH, WINDOW_HEIGHT,
                                1,
                                BlackPixel(xDisplay, DefaultScreen(xDisplay)),
                                WhitePixel(xDisplay, DefaultScreen(xDisplay)));

  xsh.flags = (USSize | PMinSize | PMaxSize);
  xsh.width = WINDOW_WIDTH;
  xsh.height = WINDOW_HEIGHT;
  xsh.min_width = WINDOW_WIDTH;
  xsh.min_height = WINDOW_HEIGHT;
  xsh.max_width = WINDOW_WIDTH;
  xsh.max_height = WINDOW_HEIGHT;

  XSetWMNormalHints(xDisplay, xWindow, &xsh);

  if(XmbTextListToTextProperty(xDisplay,
                               (char **)&cmdName,
                               1,
                               XStdICCTextStyle,
                               &w_title_property) >= Success){
    XSetWMName(xDisplay, xWindow, &w_title_property);
    XFree(w_title_property.value);
  }else{
    XStoreName(xDisplay, xWindow, cmdName);
  }

  if(XmbTextListToTextProperty(xDisplay,
                               (char **)&cmdName,
                               1,
                               XStdICCTextStyle,
                               &i_title_property) >= Success){
    XSetWMIconName(xDisplay, xWindow, &i_title_property);
    XFree(i_title_property.value);
  }else{
    XSetIconName(xDisplay, xWindow, cmdName);
  }

  /* Set Check Event */
  XSelectInput(xDisplay, xWindow,
               ExposureMask | KeyPressMask | KeyReleaseMask);

  wmDeleteWindow = XInternAtom(xDisplay, "WM_DELETE_WINDOW", False);
  XSetWMProtocols(xDisplay, xWindow, &wmDeleteWindow, 1);

#if defined (PANDORA) && defined (USE_WAITFORVSYNC)
  fbDev = open("/dev/fb0", O_RDONLY);
  if(fbDev < 0){
    printf("Couldn't open /dev/fb0 for vsync\n");
  }
#endif /* PANDORA && USE_WAITFORVSYNC */

  /* OpenGL ES */
  eglDisplay = eglGetDisplay((EGLNativeDisplayType)xDisplay);
  if(eglDisplay == EGL_NO_DISPLAY){
    printf("eglGetDisplay: error\n");
    closeWindow();
    exit(1);
  }

  if(eglInitialize(eglDisplay, NULL, NULL) == EGL_FALSE){
    printf("eglInitialize: error\n");
    closeWindow();
    exit(1);
  }

  eglBindAPI(EGL_OPENGL_ES_API);

  switch(DefaultDepth(xDisplay, DefaultScreen(xDisplay))){
  case 32:
  case 24:
    for(i = 0 ; attribute_list[i] != EGL_NONE ; i+=2){
      switch(attribute_list[i]){
      case EGL_RED_SIZE:
      case EGL_GREEN_SIZE:
      case EGL_BLUE_SIZE:
        attribute_list[i + 1] = 8;
        break;
      case EGL_BUFFER_SIZE:
        attribute_list[i + 1] = DefaultDepth(xDisplay, DefaultScreen(xDisplay));
        break;
      }
    }
    break;
  case 16:
    for(i = 0 ; attribute_list[i] != EGL_NONE ; i+=2){
      switch(attribute_list[i]){
      case EGL_RED_SIZE:
      case EGL_BLUE_SIZE:
        attribute_list[i + 1] = 5;
        break;
      case EGL_GREEN_SIZE:
        attribute_list[i + 1] = 6;
        break;
      case EGL_BUFFER_SIZE:
        attribute_list[i + 1] = DefaultDepth(xDisplay, DefaultScreen(xDisplay));
        break;
      }
    }
    break;
  default:
    printf("not support color depth (%d)\n",
           DefaultDepth(xDisplay, DefaultScreen(xDisplay)));
    closeWindow();
    exit(1);
    break;
  }

  num_config = 0;
  eglChooseConfig(eglDisplay, attribute_list, &egl_config, 1, &num_config);
  if(num_config == 0){
    printf("eglChooseConfig: error\n");
    closeWindow();
    exit(1);
  }

  eglContext = eglCreateContext(eglDisplay, egl_config, EGL_NO_CONTEXT,
                                context_attribute_list);
  if(eglContext == EGL_NO_CONTEXT){
    printf("eglCreateContext: error\n");
    closeWindow();
    exit(1);
  }

  eglSurface = eglCreateWindowSurface(eglDisplay, egl_config,
                                      (EGLNativeWindowType)xWindow, NULL);
  if(eglSurface == EGL_NO_SURFACE){
    printf("eglCreateWindowSurface: error\n");
    closeWindow();
    exit(1);
  }

  if(eglMakeCurrent(eglDisplay,
                    eglSurface, eglSurface, eglContext) != EGL_TRUE){
    printf("eglMakeCurrent: error\n");
    closeWindow();
    exit(1);
  }

  /* Back Ground Program */
  const char bg_vertex_src[] =
    "attribute vec4 a_position;    \n"
    "attribute vec2 a_texCoord;    \n"
    "varying vec2 v_texCoord;      \n"
    "                              \n"
    "void main()                   \n"
    "{                             \n"
    "   gl_Position = a_position;  \n"
    "   v_texCoord = a_texCoord;   \n"
    "}                             \n";

  const char bg_fragment_src[] =
    "precision mediump float;                           \n"
    "varying vec2 v_texCoord;                           \n"
    "uniform sampler2D s_texture;                       \n"
    "                                                   \n"
    "void main()                                        \n"
    "{                                                  \n"
    "  gl_FragColor = texture2D(s_texture, v_texCoord); \n"
    "}                                                  \n";

  bgProgram = loadProgram(bg_vertex_src, bg_fragment_src);
  if(bgProgram == 0){
    closeWindow();
    exit(1);
  }

  /* Get Attribute Locations */
  bgPositionLoc = glGetAttribLocation(bgProgram, "a_position");
  bgTexCoordLoc = glGetAttribLocation(bgProgram, "a_texCoord");
  if(bgPositionLoc < 0 || bgTexCoordLoc < 0){
    printf("glGetAttribLocation: error\n");
    closeWindow();
    exit(1);
  }

  /* Get Sampler Locations */
  bgTextureLoc = glGetUniformLocation(bgProgram, "s_texture");
  if(bgTextureLoc < 0){
    printf("glGetUniformLocation error\n");
    closeWindow();
    exit(1);
  }

  /* Back Ground Texture */
  switch(DefaultDepth(xDisplay, DefaultScreen(xDisplay))){
  case 32:
  case 24:
    /* Set Format */
    format = GL_RGB;
    //format = GL_RGBA;

    /* Set Type */
    type = GL_UNSIGNED_BYTE;
    break;
  case 16:
    /* Set Format */
    format = GL_RGB;

    /* Set Type */
    type = GL_UNSIGNED_SHORT_5_6_5;
    //type = GL_UNSIGNED_SHORT_5_5_5_1;
    //type = GL_UNSIGNED_SHORT_4_4_4_4;
    break;
  }

  /* Read Back Ground Texture */
  pixels = readTextureImage(BG_TEXTURE_FILE, &width, &height, format, type);
  if(pixels == NULL){
    printf("readTextureImage: %s read error\n", BG_TEXTURE_FILE);
    closeWindow();
    exit(1);
  }

  if(WINDOW_WIDTH > WINDOW_HEIGHT){
    wf = (GLfloat)WINDOW_HEIGHT / WINDOW_WIDTH;
    bgTexVertices[1] = bgTexVertices[7] = (1.0 - wf) / 2;
    bgTexVertices[3] = bgTexVertices[5] = wf + bgTexVertices[1];
  }else{
    wf = (GLfloat)WINDOW_WIDTH / WINDOW_HEIGHT;
    bgTexVertices[0] = bgTexVertices[2] = (1.0 - wf) / 2;
    bgTexVertices[4] = bgTexVertices[6] = wf + bgTexVertices[0];
  }

  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

  /* Generate Texture Object */
  glGenTextures(1, &bgTexture);

  /* Bind Texture Object */
  glBindTexture(GL_TEXTURE_2D, bgTexture);

  /* Load Texture */
  glTexImage2D(GL_TEXTURE_2D, 0, format,
               width, height, 0, format, type, pixels);

  /* Set Filtering Mode */
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

  free(pixels);

  /* Fps Char Texture */
  format = GL_RGBA;
  type = GL_UNSIGNED_SHORT_4_4_4_4;

  /* Read Fps Char Texture */
  pixels = readTextureImage(FPS_TEXTURE_FILE, &width, &height, format, type);
  if(pixels == NULL){
    printf("readTextureImage: %s read error\n", FPS_TEXTURE_FILE);
    closeWindow();
    exit(1);
  }

  glPixelStorei(GL_UNPACK_ALIGNMENT, 2);

  /* Generate Texture Object */
  glGenTextures(1, &fpsTexture);

  /* Bind Texture Object */
  glBindTexture(GL_TEXTURE_2D, fpsTexture);

  /* Load Texture */
  glTexImage2D(GL_TEXTURE_2D, 0, format,
               width, height, 0, format, type, pixels);

  /* Set Filtering Mode */
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

  free(pixels);

  /* Object-1 Program */
  const char vertex_src[] =
    "attribute vec4        position;    \n"
    "varying mediump vec2  pos;         \n"
    "uniform vec4          offset;      \n"
    "                                   \n"
    "void main()                        \n"
    "{                                  \n"
    "  gl_Position = position + offset; \n"
    "  pos = position.xy;               \n"
    "}                                  \n";

  const char fragment_src[] =
    "varying mediump vec2    pos;                             \n"
    "uniform mediump float   phase;                           \n"
    "uniform mediump vec4    color;                           \n"
    "                                                         \n"
    "void main()                                              \n"
    "{                                                        \n"
    "  gl_FragColor = color *                                 \n"
    "    cos(50.0 * sqrt(pos.x * pos.x + 1.5 * pos.y * pos.y) \n"
    "        + atan(pos.y, pos.x) - phase);                   \n"
    "}                                                        \n";

  object1Program = loadProgram(vertex_src, fragment_src);
  if(object1Program == 0){
    closeWindow();
    exit(1);
  }

  /* Get Attribute Locations */
  positionLoc = glGetAttribLocation(object1Program, "position");
  if(positionLoc < 0){
    printf("glGetAttribLocation: error\n");
    closeWindow();
    exit(1);
  }

  /* Get Sampler Locations */
  phaseLoc = glGetUniformLocation(object1Program, "phase");
  offsetLoc = glGetUniformLocation(object1Program, "offset");
  colorLoc = glGetUniformLocation(object1Program, "color");
  if(phaseLoc < 0 || offsetLoc < 0 || colorLoc < 0){
    printf("glGetUniformLocation error\n");
    closeWindow();
    exit(1);
  }

  /* Triangle Program */
  const char tri_vertex_src[] =
    "attribute vec4 vPosition;    \n"
    "void main()                  \n"
    "{                            \n"
    "   gl_Position = vPosition;  \n"
    "}                            \n";

  const char tri_fragment_src[] =
    "precision mediump float;                    \n"
    "void main()                                 \n"
    "{                                           \n"
    "  gl_FragColor = vec4(1.0, 0.5, 0.5, 1.0);  \n"
    "}                                           \n";

  triangleProgram = loadProgram(tri_vertex_src, tri_fragment_src);
  if(triangleProgram == 0){
    closeWindow();
    exit(1);
  }

  /* Get Attribute Locations */
  triangleLoc = glGetAttribLocation(triangleProgram, "vPosition");
  if(triangleLoc < 0){
    printf("glGetAttribLocation: error\n");
    closeWindow();
    exit(1);
  }

  glClearColor(0.0, 0.0, 0.0, 0.0);

  nubProbe();

  if(nubNums != NUB_MAX){
    printf("warning: nub found %d/%d\n", nubNums, NUB_MAX);
  }

  /* Map Window */
  XMapWindow(xDisplay, xWindow);

  /* Full Screen */
  if(DisplayWidth(xDisplay, DefaultScreen(xDisplay)) == WINDOW_WIDTH
  && DisplayHeight(xDisplay, DefaultScreen(xDisplay)) == WINDOW_HEIGHT){
    nws_fullscreen = XInternAtom(xDisplay, "_NET_WM_STATE_FULLSCREEN", False);
    nws = XInternAtom(xDisplay, "_NET_WM_STATE", False);

    xev.xclient.type = ClientMessage;
    xev.xclient.serial = 0;
    xev.xclient.send_event = True;
    xev.xclient.display = xDisplay;
    xev.xclient.window = xWindow;
    xev.xclient.message_type = nws;
    xev.xclient.format = 32;
    xev.xclient.data.l[0] = 1;
    xev.xclient.data.l[1] = nws_fullscreen;
    xev.xclient.data.l[2] = 0;
    xev.xclient.data.l[3] = 0;
    xev.xclient.data.l[4] = 0;

    XSendEvent(xDisplay, DefaultRootWindow(xDisplay), False,
               SubstructureNotifyMask, &xev);
  }

  XSync(xDisplay, False);

  /* Event Loop */
  eventLoop();

  closeWindow();

  exit(0);
}
