/*
Copyright (c) 2011 Sano Yukihiko

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/
#define XTGV_CURRENT_VERSION "0.8.0_alpha1"
char *XtgvVersion = XTGV_CURRENT_VERSION;

#include "xtgv.h"
#include "xtgv_char.h"

void xtgvReadOptionConfig(char *path)
{
    FILE *fp;
    char conf_buf[XTGV_OPTION_CONFIG_LINE_MAX + 1];
    int ret;
    char *ptr;

    /* Open File */
    fp = fopen(path, "r");
    if(fp == NULL){
        return;
    }

    /* Read File */
    while(fgets(conf_buf, XTGV_OPTION_CONFIG_LINE_MAX, fp)){
        ptr = strrchr(conf_buf, '\n');
        if(ptr != NULL){
            ptr[0] = '\0';
        }
        ptr = strrchr(conf_buf, '\r');
        if(ptr != NULL){
            ptr[0] = '\0';
        }
        if(conf_buf[0] == '#'){
            continue;
        }

        for(ptr = conf_buf ; ptr[0] != '\0' && ptr[0] == ' ' ; ptr++);

        if(ptr[0] != '-'){
            continue;
        }

        if((strncmp(ptr, "-fn", 3)) == 0){
            ptr = ptr + 3;
            for( ; ptr[0] != '\0' && ptr[0] == ' ' ; ptr++);
            if(ptr[0] == '\0'){
                continue;
            }

            if(strlen(ptr) > XTGV_FONT_NAME_OPTION_MAX){
                strncpy(FontNameOptConf, ptr, XTGV_FONT_NAME_OPTION_MAX);
            }else{
                strcpy(FontNameOptConf, ptr);
            }
        }
        else
        if((strncmp(ptr, "-fg", 3)) == 0){
            ptr = ptr + 3;
            for( ; ptr[0] != '\0' && ptr[0] == ' ' ; ptr++);
            if(ptr[0] == '\0'){
                continue;
            }

            if(strlen(ptr) > XTGV_COLOR_NAME_OPTION_MAX){
                strncpy(FgColorNameOptConf, ptr, XTGV_COLOR_NAME_OPTION_MAX);
            }else{
                strcpy(FgColorNameOptConf, ptr);
            }
        }
        else
        if((strncmp(ptr, "-bg", 3)) == 0){
            ptr = ptr + 3;
            for( ; ptr[0] != '\0' && ptr[0] == ' ' ; ptr++);
            if(ptr[0] == '\0'){
                continue;
            }

            if(strlen(ptr) > XTGV_COLOR_NAME_OPTION_MAX){
                strncpy(BgColorNameOptConf, ptr, XTGV_COLOR_NAME_OPTION_MAX);
            }else{
                strcpy(BgColorNameOptConf, ptr);
            }
        }
        else
        if((strncmp(ptr, "-p", 2)) == 0){
            ptr = ptr + 2;
            for( ; ptr[0] != '\0' && ptr[0] == ' ' ; ptr++);
            if(ptr[0] == '\0'){
                continue;
            }

            ret = sscanf(ptr, "%d", &CurrentPageOpt);
            if(ret != 1){
                CurrentPageOpt = 1;
            }
        }
        else
        if((strncmp(ptr, "-s", 2)) == 0){
            ptr = ptr + 2;
            for( ; ptr[0] != '\0' && ptr[0] == ' ' ; ptr++);
            if(ptr[0] == '\0'){
                continue;
            }

            ret = sscanf(ptr, "%d %d", &LinesOfPageOpt, &CharsOfLineOpt);
            if(ret != 2){
                LinesOfPageOpt = 0;
                CharsOfLineOpt = 0;
                continue;
            }
            if(LinesOfPageOpt < XTGV_LINES_OF_PAGE_MIN){
                LinesOfPageOpt = 0;
            }
            if(CharsOfLineOpt < XTGV_CHARS_OF_LINE_MIN){
                CharsOfLineOpt = 0;
            }
        }
#ifdef USE_XFT
        else
        if((strncmp(ptr, "-a", 2)) == 0){
            XftUseFlag = 1;
        }
#endif /* USE_XFT */
        else
        if((strncmp(ptr, "-t", 2)) == 0){
            TagCutFlagOpt = 1;
        }
    }

    /* Close File */
    fclose(fp);

    return;
}

#if defined(USE_IMLIB2) || defined(USE_IMLIB)
void xtgvImageDataListFree(void)
{
    int i;

    if(ImageDataList.ptr != NULL){
        for(i = 0 ; i < ImageDataList.exist ; i++){
            if(ImageDataList.ptr[i].path != NULL){
                free(ImageDataList.ptr[i].path);
            }
            ImageDataList.ptr[i].path = NULL;

            if(ImageDataList.ptr[i].pixmap > 0){
                XFreePixmap(DpyOfX, ImageDataList.ptr[i].pixmap);
            }
            ImageDataList.ptr[i].pixmap = 0;
        }

        free(ImageDataList.ptr);
    }
    ImageDataList.ptr = NULL;

    ImageDataList.max = 0;
    ImageDataList.exist = 0;

    return;
}
#endif /* USE_IMLIB2 || USE_IMLIB */

int xtgvHtmlTagCheck(unsigned char *ptr, int x, int y, int flag)
{
#if defined(USE_IMLIB2) || defined(USE_IMLIB)
    int i;
    int ip;
    int cnt;
    int sx;
    int sy;
    int c_width;
    int i_width;
    int i_height;
    int use_line;
    double aspect;
    char *sp;
    char *start;
    char *end;
    char *wk_path;
    XtgvIDE *wk_ide;
    Pixmap wk_pixmap;
#if defined(USE_IMLIB2)
    Imlib_Image imlib_image;
#elif defined(USE_IMLIB)
    ImlibImage *imlib_image;
    Pixmap tmp_pixmap;
#endif /* USE_IMLIB */

    sp = (char *)ptr;
    aspect = 0;

    /* Check Image Tag */
    if(strncmp(sp, "<img ", 5) != 0 &&
       strncmp(sp, "<IMG ", 5) != 0){
        return 0;
    }

    for(sp = sp + 5 ; sp[0] != '\0' ; sp++){
        if(sp[0] != ' '){
            break;
        }
    }
    if(strncmp(sp, "src", 3) != 0 &&
       strncmp(sp, "SRC", 3) != 0){
        return 0;
    }
    sp = sp + 3;
    start = strchr(sp, '"');
    if(start == NULL){
        return 0;
    }
    start++;
    sp = start;
    end = strchr(sp, '"');
    if(end == NULL){
        return 0;
    }

    cnt = end - start;
    wk_path = (char *)malloc(cnt + 1);
    if(wk_path == NULL){
        return 0;
    }

    memset(wk_path, 0x00, cnt + 1);
    memcpy(wk_path, start, (size_t)cnt);

    for(i = 0, ip = -1 ; i < ImageDataList.exist ; i++){
        if(strcmp(wk_path, ImageDataList.ptr[i].path) == 0){
            ip = i;
            break;
        }
    }

    if(ip != -1){
        free(wk_path);
        if(flag == XTGV_DISPLAY_PAGE_ON){
            XCopyArea(DpyOfX, ImageDataList.ptr[ip].pixmap,
                      MainPixmap, FgColorGC, 0, 0,
                      ImageDataList.ptr[ip].width,
                      ImageDataList.ptr[ip].height,
                      ImageDataList.ptr[ip].x,
                      ImageDataList.ptr[ip].y);
        }
        return ImageDataList.ptr[ip].line;
    }

#if defined(USE_IMLIB2)
    imlib_image = imlib_load_image(wk_path);
    if(imlib_image == NULL){
        free(wk_path);
        return 0;
    }

    imlib_context_set_image(imlib_image);

    /* After imlib_context_set_image() ? */
    i_width = imlib_image_get_width();
    i_height = imlib_image_get_height();

#elif defined(USE_IMLIB)
    if(ImlibDataExt == NULL){
        free(wk_path);
        return 0;
    }

    imlib_image = Imlib_load_image(ImlibDataExt, wk_path);
    if(imlib_image == NULL){
        fprintf(stderr,"Imlib_load_image failed.\n");
        free(wk_path);
        return 0;
    }

    i_width = imlib_image->rgb_width;
    i_height = imlib_image->rgb_height;

#endif /* USE_IMLIB */

    if(i_width > (MainPixmapW - FontDataInfo.ruby_box_w * 2)){
        /* Image Smaller */
        aspect = (double)i_height / i_width;
        i_width = (MainPixmapW - FontDataInfo.ruby_box_w * 2);
        i_height = i_width * aspect;
    }

    if(i_height > MainPixmapH){
        /* Image Smaller */
        aspect = (double)i_width / i_height;
        i_height = MainPixmapH;
        i_width = i_height * aspect;
    }

    /* Create Pixmap */
    wk_pixmap = XCreatePixmap(DpyOfX, DefaultRootWindow(DpyOfX),
                              i_width, i_height,
                              DefaultDepth(DpyOfX, XTGV_GET_SCREEN));
    if(wk_pixmap <= 0){
#if defined(USE_IMLIB2)
        imlib_free_image();
#elif defined(USE_IMLIB)
        Imlib_kill_image(ImlibDataExt, imlib_image);
#endif /* USE_IMLIB */
        free(wk_path);
        return 0;
    }

#if defined(USE_IMLIB2)
    imlib_context_set_drawable(wk_pixmap);

    if(aspect == 0){
        imlib_render_image_on_drawable(0, 0);
    }else{
        imlib_render_image_on_drawable_at_size(0, 0, i_width, i_height);
    }

    /* free of imlib_image */
    imlib_free_image();

#elif defined(USE_IMLIB)
    Imlib_render(ImlibDataExt, imlib_image, i_width, i_height);
    tmp_pixmap = Imlib_move_image(ImlibDataExt, imlib_image);
    if(tmp_pixmap <= 0){
        Imlib_kill_image(ImlibDataExt, imlib_image);
        free(wk_path);
        return 0;
    }

    XCopyArea(DpyOfX, tmp_pixmap, wk_pixmap, FgColorGC,
              0, 0, i_width, i_height,
              0, 0);

    Imlib_free_pixmap(ImlibDataExt, tmp_pixmap);
    Imlib_kill_image(ImlibDataExt, imlib_image);

#endif /* USE_IMLIB */

    c_width = FontDataInfo.box_w + FontDataInfo.ruby_box_w;
    use_line = (i_width + FontDataInfo.ruby_box_w + c_width - 1) / c_width;

    /* ImageDataList Add */
    if(ImageDataList.ptr == NULL){
        ImageDataList.ptr = (XtgvIDE *)malloc(sizeof(XtgvIDE) * XTGV_IMAGE_MAX);
        if(ImageDataList.ptr == NULL){
            perror("malloc");
            free(wk_path);
            XFreePixmap(DpyOfX, wk_pixmap);
            return 0;
        }
        memset(ImageDataList.ptr, 0, sizeof(XtgvIDE) * XTGV_IMAGE_MAX);
        ImageDataList.max = XTGV_IMAGE_MAX;
    }
    if(ImageDataList.exist >= ImageDataList.max){
        wk_ide = (XtgvIDE *)malloc(sizeof(XtgvIDE) *
                                  (ImageDataList.max + XTGV_IMAGE_MAX));
        if(wk_ide == NULL){
            perror("malloc");
            free(wk_path);
            XFreePixmap(DpyOfX, wk_pixmap);
            return 0;
        }
        memset(wk_ide, 0, sizeof(XtgvIDE) *
               (ImageDataList.max + XTGV_IMAGE_MAX));
        ImageDataList.max = ImageDataList.max + XTGV_IMAGE_MAX;
        /* Copy */
        for(i = 0 ; i < ImageDataList.exist ; i++){
            wk_ide[i].path = ImageDataList.ptr[i].path;
            wk_ide[i].x = ImageDataList.ptr[i].x;
            wk_ide[i].y = ImageDataList.ptr[i].y;
            wk_ide[i].width = ImageDataList.ptr[i].width;
            wk_ide[i].height = ImageDataList.ptr[i].height;
            wk_ide[i].line = ImageDataList.ptr[i].line;
            wk_ide[i].pixmap = ImageDataList.ptr[i].pixmap;
        }

        free(ImageDataList.ptr);
        ImageDataList.ptr = wk_ide;
    }

    ImageDataList.ptr[ImageDataList.exist].path = wk_path;
    ImageDataList.ptr[ImageDataList.exist].x = FontDataInfo.ruby_box_w;
    ImageDataList.ptr[ImageDataList.exist].y = 0;
    ImageDataList.ptr[ImageDataList.exist].width = i_width;
    ImageDataList.ptr[ImageDataList.exist].height = i_height;
    ImageDataList.ptr[ImageDataList.exist].line = use_line;
    ImageDataList.ptr[ImageDataList.exist].pixmap = wk_pixmap;

    sx = x - i_width + FontDataInfo.box_w;
    if(sx < FontDataInfo.ruby_box_w){
        /* New Page */
        sx = MainPixmapW - i_width - FontDataInfo.ruby_box_w;
        //sy = (MainPixmapH - i_height) / 2;
        sy = 0;
        ImageDataList.ptr[ImageDataList.exist].x = sx;
        ImageDataList.ptr[ImageDataList.exist].y = sy;
        ImageDataList.exist++;
        return -1;
    }

    ImageDataList.ptr[ImageDataList.exist].x = sx;

    sy = y;
    if((sy + i_height) > MainPixmapH){
        use_line++;
        sx = sx - c_width;
        if(sx < FontDataInfo.ruby_box_w){
            /* New Page */
            sx = MainPixmapW - i_width - FontDataInfo.ruby_box_w;
            //sy = (MainPixmapH - i_height) / 2;
            sy = 0;
            ImageDataList.ptr[ImageDataList.exist].x = sx;
            ImageDataList.ptr[ImageDataList.exist].y = sy;
            ImageDataList.exist++;
            return -1;
        }
        //sy = (MainPixmapH - i_height) / 2;
        sy = 0;
        ImageDataList.ptr[ImageDataList.exist].x = sx;
        ImageDataList.ptr[ImageDataList.exist].y = sy;
        ImageDataList.ptr[ImageDataList.exist].line = use_line;
    }

    ImageDataList.ptr[ImageDataList.exist].y = sy;

    if(flag == XTGV_DISPLAY_PAGE_ON){
        XCopyArea(DpyOfX, ImageDataList.ptr[ImageDataList.exist].pixmap,
                  MainPixmap, FgColorGC, 0, 0,
                  ImageDataList.ptr[ImageDataList.exist].width,
                  ImageDataList.ptr[ImageDataList.exist].height,
                  ImageDataList.ptr[ImageDataList.exist].x,
                  ImageDataList.ptr[ImageDataList.exist].y);
    }

    ImageDataList.exist++;

    return use_line;

#endif /* USE_IMLIB2 || USE_IMLIB */
    /* Check Ruby Tag */

    return 0;
}

unsigned char *xtgvHtmlTagDelete(unsigned char *ptr)
{
    unsigned char *sp;
    unsigned char *ep;
    unsigned char *wp;

    sp = ptr;

    if(sp[0] != '<'){
        return NULL;
    }

    sp++;
    if(sp[0] == '\0'){
        return NULL;
    }

    if(isalpha(sp[0]) != 0 || sp[0] == '/'){
        for(wp = sp, ep = NULL ; wp[0] != '\0' ; wp++){
            if(wp[0] == '>'){
                ep = wp;
                break;
            }
        }
        if(ep == NULL){
            return NULL;
        }
        ep++;
        return ep;
    }else{
        if(sp[0] != '!'){
            return NULL;
        }
        sp++;
        if(sp[0] == '\0'){
            return NULL;
        }

        if(sp[0] != '-'){
            if(strlen((char *)sp) > 7){
                /* <!DOCTYPE */
                if(strncmp((char *)sp, "doctype ", 8) != 0 &&
                   strncmp((char *)sp, "DOCTYPE ", 8) != 0){
                    return NULL;
                }
                if(sp[8] == '\0'){
                    return NULL;
                }
                if(sp[8] == '>' || sp[8] == '<'){
                    return NULL;
                }
                sp = sp + 8;
            }else{
                return NULL;
            }
        }else{
            /* <!-- */
            sp++;
            if(sp[0] == '\0'){
                return NULL;
            }
            if(sp[0] != '-'){
                return NULL;
            }
            sp++;
            if(sp[0] == '\0'){
                return NULL;
            }
            if(sp[0] == '-' || sp[0] == '>' || sp[0] == '<'){
                return NULL;
            }
        }
        sp++;
        if(sp[0] == '\0'){
            return NULL;
        }
        for( ; sp[0] != '\0' ; sp++){
            for(wp = sp, ep = NULL ; wp[0] != '\0' ; wp++){
                if(wp[0] == '>'){
                    ep = wp;
                    break;
                }
            }
            if(ep == NULL){
                return NULL;
            }
            wp = ep;
            wp-=2;
            if(wp[0] == '-' && wp[1] == '-'){
                ep++;
                return ep;
            }else{
                sp = ep;
            }
        }
        return NULL;
    }
}

void xtgvDrawString(unsigned char *ptr, int len, int x, int y, int dst)
{
    int sx;
    int sy;
    Drawable d;
    XRectangle dummy, bb;
#ifdef USE_XFT
    XftDraw *dxft;
#endif /* USE_XFT */

    switch(dst){
    case 0:
        d = MainPixmap;
#ifdef USE_XFT
        dxft = DrawXft;
#endif /* USE_XFT */
        break;
    case 1:
        d = ConvPixmap;
#ifdef USE_XFT
        dxft = ConvXft;
#endif /* USE_XFT */
        break;
    default:
        d = MainPixmap;
#ifdef USE_XFT
        dxft = DrawXft;
#endif /* USE_XFT */
        break;
    }

#ifdef USE_XFT
    if(XftUseFlag == 1){
        sx = x + FontDataInfo.gap_x;

        if(len == 1){
            sy = y + FontDataInfo.gap_y + FontDataInfo.char_h * 8 / 10;

            XftDrawString8(dxft, &ColorXft, FontXft,
                           sx,
                           sy,
                           (FcChar8 *)ptr, len);
        }else{
            sy = y + FontDataInfo.gap_y + FontDataInfo.char_h * 9 / 10;

            if(CurrentLocale == XTGV_UTF){
                XftDrawStringUtf8(dxft, &ColorXft, FontXft,
                                  sx,
                                  sy,
                                  (FcChar8 *)ptr, len);
            }else{
size_t cnum;
size_t in;
size_t out;
unsigned char convp[XTGV_CHAR_LEN];
char *utf;

                in = len;
                out = XTGV_CHAR_LEN - 1;
                memset(convp, 0, XTGV_CHAR_LEN);
                utf = (char *)convp;

                cnum = iconv(IconvCD, (char **)&ptr, &in, &utf, &out);
                if(cnum == -1){
                    return;
                }

                XftDrawStringUtf8(dxft, &ColorXft, FontXft,
                                  sx,
                                  sy,
                                  (FcChar8 *)convp, out);
            }
        }
    }else{
#endif /* USE_XFT */
        XmbTextExtents(FontSetXCF, (char *)ptr, len, &dummy, &bb);

        sx = x + FontDataInfo.gap_x + bb.x;
        //sy = y + FontDataInfo.gap_y + FontDataInfo.char_h * 8 / 10;
        sy = y + FontDataInfo.gap_y + bb.height - (bb.y + bb.height);

        XmbDrawImageString(DpyOfX, d,
                           FontSetXCF,
                           FgColorGC,
                           sx,
                           sy,
                           (char *)ptr, len);
#ifdef USE_XFT
    }
#endif /* USE_XFT */

    return;
}

int xtgvCreateXImage(XImage **d, XImage **s, unsigned char *ptr, int len)
{
    XFillRectangle(DpyOfX, ConvPixmap,
                   BgColorGC,
                   0, 0,
                   FontDataInfo.box_w,
                   FontDataInfo.box_h);

    *d = XGetImage(DpyOfX, ConvPixmap,
                   0, 0,
                   FontDataInfo.box_w,
                   FontDataInfo.box_h,
                   AllPlanes, XYPixmap);
    if(*d == NULL){
        return 1;
    }

    xtgvDrawString(ptr, len, 0, 0, 1);

    *s = XGetImage(DpyOfX, ConvPixmap,
                   0, 0,
                   FontDataInfo.box_w,
                   FontDataInfo.box_h,
                   AllPlanes, XYPixmap);
    if(*s == NULL){
        if(*d != NULL){
            XDestroyImage(*d);
        }
        *d = NULL;
        return 1;
    }

    return 0;
}

XtgvCCE *xtgvConvTypeCheck(unsigned char *str)
{
    int i;
    int ret;
    int ck_num;
    int ccl_num;

    for(i = 0 ; i < ConvCharList.exist ; i++){
        ck_num = xtgvChrNum(str);
        ccl_num = ConvCharList.ptr[i].num;
        if(ck_num < ccl_num){
            continue;
        }
        ret = memcmp(str, ConvCharList.ptr[i].str, (size_t)ccl_num);
        if(ret == 0){
            return &(ConvCharList.ptr[i]);
        }
    }

    return NULL;
}

void xtgvInit(void)
{
    CmdName = NULL;
    CurrentLang = NULL;
    CurrentLocale = XTGV_UTF;

    FontNameOpt = NULL;
    FgColorNameOpt = NULL;
    BgColorNameOpt = NULL;
    FileNameOpt = NULL;
    CurrentPageOpt = 1;
    TagCutFlagOpt = 0;
    LinesOfPageOpt = 0;
    CharsOfLineOpt = 0;

    LinesOfPage = XTGV_LINES_OF_PAGE;
    CharsOfLine = XTGV_CHARS_OF_LINE;

    ReadData = NULL;

    FgColorName = XTGV_FG_COLOR;
    BgColorName = XTGV_BG_COLOR;

    memset(FontNameOptConf, 0, XTGV_FONT_NAME_OPTION_MAX + 1);
    memset(FgColorNameOptConf, 0, XTGV_COLOR_NAME_OPTION_MAX + 1);
    memset(BgColorNameOptConf, 0, XTGV_COLOR_NAME_OPTION_MAX + 1);

    FontSetXCF = NULL;
    DpyOfX = NULL;
    BgColorGC = 0;
    FgColorGC = 0;
    MainPixmap = 0;
    ConvPixmap = 0;
#ifdef USE_XFT
    IconvCD = NULL;
    XftUseFlag = 0;
    FontXft = NULL;
    DrawXft = NULL;
    ConvXft = NULL;
    MainWinXft = NULL;
    FontNameXft = XTGV_XFT_FONT_NAME;
#endif /* USE_XFT */
    FontNameXCF = XTGV_XCF_FONT_NAME;

    PageDataList.page_data = NULL;
    PageDataList.max = 0;
    PageDataList.exist = 0;
    PageDataList.complete = XTGV_FALSE;
    PageDataList.current = 0;

    ConvCharList.ptr = NULL;
    ConvCharList.max = 0;
    ConvCharList.exist = 0;

#if defined(USE_IMLIB2) || defined(USE_IMLIB)
    ImageDataList.ptr = NULL;
    ImageDataList.max = 0;
    ImageDataList.exist = 0;
#endif /* USE_IMLIB2 || USE_IMLIB */

    return;
}

void xtgvExit(void)
{
    if(DpyOfX != NULL){
#ifdef USE_XFT
        if(XftUseFlag == 1){
            if(IconvCD != NULL){
                iconv_close(IconvCD);
            }
            IconvCD = NULL;
            if(FontXft != NULL){
                XftFontClose(DpyOfX, FontXft);
            }
            FontXft = NULL;
            if(DrawXft != NULL){
                XftDrawDestroy(DrawXft);
            }
            DrawXft = NULL;
            if(ConvXft != NULL){
                XftDrawDestroy(ConvXft);
            }
            ConvXft = NULL;
            if(MainWinXft != NULL){
                XftDrawDestroy(MainWinXft);
            }
            MainWinXft = NULL;
        }else{
#endif /* USE_XFT */
            if(FontSetXCF != NULL){
                XFreeFontSet(DpyOfX, FontSetXCF);
            }
#ifdef USE_XFT
        }
#endif /* USE_XFT */

#if defined(USE_IMLIB2) || defined(USE_IMLIB)
        xtgvImageDataListFree();
#endif /* USE_IMLIB2 || USE_IMLIB */

        if(MainPixmap > 0){
            XFreePixmap(DpyOfX, MainPixmap);
        }
        MainPixmap = 0;

        if(ConvPixmap > 0){
            XFreePixmap(DpyOfX, ConvPixmap);
        }
        ConvPixmap = 0;

        xtgvConvCharListFree();

        if(FgColorGC > 0){
            XFreeGC(DpyOfX, FgColorGC);
        }

        if(BgColorGC > 0){
            XFreeGC(DpyOfX, BgColorGC);
        }
        XDestroyWindow(DpyOfX, MainWin);
        XCloseDisplay(DpyOfX);
    }

    xtgvPageDataListFree();

    if(ReadData != NULL){
        free(ReadData);
    }
    ReadData = NULL;

    return;
}

void xtgvConvCharListFree(void)
{
    int i;

    if(ConvCharList.ptr != NULL){
        for(i = 0 ; i < ConvCharList.exist ; i++){
            if(ConvCharList.ptr[i].conv_image != NULL){
                XDestroyImage(ConvCharList.ptr[i].conv_image);
            }
            ConvCharList.ptr[i].conv_image = NULL;
        }

        free(ConvCharList.ptr);
    }
    ConvCharList.ptr = NULL;

    ConvCharList.max = 0;
    ConvCharList.exist = 0;

    return;
}

void xtgvPageDataListFree(void)
{
    //int i;

    if(PageDataList.page_data != NULL){
        //for(i = 0 ; i < PageDataList.exist ; i++){
        //}
        free(PageDataList.page_data);
    }
    PageDataList.page_data = NULL;

    PageDataList.max = 0;
    PageDataList.exist = 0;
    PageDataList.complete = XTGV_FALSE;
    PageDataList.current = 0;

    return;
}

int xtgvCheckOption(int ac, char **av, int flag)
{
    int i;
    int ret;

    for(i = 1 ; i < ac ; i++){
        if((strcmp(av[i], "-h")) == 0){
            printf("Usage: %s", CmdName);
            printf(" [-fn font]");
#ifdef USE_XFT
            printf(" [-a]");
#endif /* USE_XFT */
            printf(" [-fg color]");
            printf(" [-bg color]");
            printf(" [-p number]");
            printf(" [-s h v]");
            printf(" [-t]");
            printf(" [-v]");
            printf(" [-h]");
            printf(" [file]\n");
            return 1;
        }
        else
        if((strcmp(av[i], "-fn")) == 0){
            if((i + 1) >= ac){
                fprintf(stderr, "Option -h display help.\n");
                return 1;
            }
            FontNameOpt = av[++i];
        }
        else
        if((strcmp(av[i], "-fg")) == 0){
            if((i + 1) >= ac){
                fprintf(stderr, "Option -h display help.\n");
                return 1;
            }
            FgColorNameOpt = av[++i];
        }
        else
        if((strcmp(av[i], "-bg")) == 0){
            if((i + 1) >= ac){
                fprintf(stderr, "Option -h display help.\n");
                return 1;
            }
            BgColorNameOpt = av[++i];
        }
        else
        if((strcmp(av[i], "-p")) == 0){
            if((i + 1) >= ac){
                fprintf(stderr, "Option -h display help.\n");
                return 1;
            }
            ret = sscanf(av[++i], "%d", &CurrentPageOpt);
            if(ret != 1){
                fprintf(stderr, "Option -h display help.\n");
                return 1;
            }
        }
        else
        if((strcmp(av[i], "-s")) == 0){
            if((i + 2) >= ac){
                fprintf(stderr, "Option -h display help.\n");
                return 1;
            }
            ret = sscanf(av[++i], "%d", &LinesOfPageOpt);
            if(ret != 1){
                fprintf(stderr, "Option -h display help.\n");
                return 1;
            }
            if(LinesOfPageOpt < XTGV_LINES_OF_PAGE_MIN){
                LinesOfPageOpt = 0;
            }
            ret = sscanf(av[++i], "%d", &CharsOfLineOpt);
            if(ret != 1){
                fprintf(stderr, "Option -h display help.\n");
                return 1;
            }
            if(CharsOfLineOpt < XTGV_CHARS_OF_LINE_MIN){
                CharsOfLineOpt = 0;
            }
        }
#ifdef USE_XFT
        else
        if((strcmp(av[i], "-a")) == 0){
            XftUseFlag = 1;
        }
#endif /* USE_XFT */
        else
        if((strcmp(av[i], "-t")) == 0){
            TagCutFlagOpt = 1;
        }
        else
        if((strcmp(av[i], "-v")) == 0){
            printf("xtgv - X Tate Gaki Viewer");
            printf("    version %s\n", XtgvVersion);
            return 1;
        }
        else{
            if(av[i][0] == '-'){
                printf("Option -h display help\n");
                return 1;
            }
            if(FileNameOpt != NULL){
                continue;
            }
            if(flag == 1){
                FileNameOpt = av[i];
            }
        }
    }

    return 0;
}

void xtgvPageReDraw(void)
{
    int c_page;
    int a_page;
    int len;
    int sy;
    char page_str[16];
    XRectangle dummy, bb;

    XClearWindow(DpyOfX, MainWin);
    XCopyArea(DpyOfX, MainPixmap, MainWin, FgColorGC,
              0, 0, MainPixmapW, MainPixmapH,
              0, FontDataInfo.box_h / 2);

    /* Page */
    c_page = PageDataList.current;
    a_page = PageDataList.exist;

    if(c_page > 9999){
        c_page = 9999;
    }

    if(a_page > 9999){
        a_page = 9999;
    }

    if(PageDataList.complete == XTGV_TRUE){
        sprintf(page_str, " [%4d/%4d]", c_page, a_page);
    }else{
        sprintf(page_str, " [%4d/----]", c_page);
    }

    len = strlen(page_str);

#ifdef USE_XFT
    if(XftUseFlag == 1){
        sy = MainWinH - FontDataInfo.char_h * 2 / 10 - FontDataInfo.gap_y;
        XftDrawString8(MainWinXft, &ColorXft, FontXft,
                       0,
                       sy,
                       (FcChar8 *)page_str, len);
    }else{
#endif /* USE_XFT */
        XmbTextExtents(FontSetXCF, page_str, len, &dummy, &bb);
        sy = MainWinH - FontDataInfo.char_h + bb.height - (bb.y + bb.height);
        XmbDrawImageString(DpyOfX, MainWin,
                           FontSetXCF,
                           FgColorGC,
                           0,
                           sy,
                           page_str, len);
#ifdef USE_XFT
    }
#endif /* USE_XFT */

    sy = FontDataInfo.box_h / 2 + MainPixmapH - 1;
    XDrawLine(DpyOfX, MainWin, FgColorGC, 0, sy, MainWinW - 1, sy);

    return;
}

void xtgvEventLoop(void)
{
    int e_loop = 1;
    int ret_key;
    char string[10];
    XEvent event;
    KeySym key;
    XComposeStatus cs;

    while(e_loop){
        XNextEvent(DpyOfX, &event);
        switch(event.type){
        case DestroyNotify:
            e_loop = 0;
            break;
        case UnmapNotify:
            e_loop = 0;
            break;
        case Expose:
            xtgvPageReDraw();
            break;
        case KeyPress:
            for(ret_key = 0 ; ret_key < 10 ; ret_key++){
                string[ret_key] = 0x00;
            }

            ret_key = XLookupString((XKeyEvent *)&event, string, 1, &key, &cs);

            switch(key){
            case XK_Home:
            case XK_KP_Home:
                /* Top Page */
                if(xtgvGotoPage(1) != 0){
                    e_loop = 0;
                }
                break;
            case XK_End:
            case XK_KP_End:
                /* End Page */
                if(xtgvGotoPage(-1) != 0){
                    e_loop = 0;
                }
                break;
            case XK_Page_Up:
            case XK_KP_Page_Up:
                /* Pre Page */
                if(PageDataList.current > 1){
                    if(xtgvGotoPage(PageDataList.current - 1) != 0){
                        e_loop = 0;
                    }
                }
                break;
            case XK_Page_Down:
            case XK_KP_Page_Down:
                /* Next Page */
                if(xtgvGotoPage(PageDataList.current + 1) != 0){
                    e_loop = 0;
                }
                break;
            case 'q':
                e_loop = 0;
                break;
            case 'b':
                /* Pre Page */
                if(((XKeyEvent *)&event)->state & ControlMask){
                    if(PageDataList.current > 1){
                        if(xtgvGotoPage(PageDataList.current - 1) != 0){
                            e_loop = 0;
                        }
                    }
                }
                break;
            case 'f':
                /* Next Page */
                if(((XKeyEvent *)&event)->state & ControlMask){
                    if(xtgvGotoPage(PageDataList.current + 1) != 0){
                        e_loop = 0;
                    }
                }
                break;
            }
            break;
        }
    }
    return;
}

unsigned char *xtgvLayoutPage(unsigned char *start, unsigned char *end, int flag)
{
    unsigned char *ptr;
    int i;
    int j;
    int sx;
    int sy;
    int cx;
    int cy;
    int dx;
    int dy;
    int len;
    int ret;
    int conv_type;
    XtgvCCE *wk_cce;
    XImage *wk_image;
    XImage *ascii_image;
    unsigned long pixel;
    int conv_failed;
    int ascii_half;
    int bg_sx;
    int bg_sy;
    int bg_ex;
    int bg_ey;
    int use_line;
    unsigned char *r_ptr;

    ptr = start;

    if(ptr == NULL){
        return NULL;
    }

    if(flag == XTGV_DISPLAY_PAGE_ON){
        //XClearWindow(DpyOfX, MainPixmap);
        XFillRectangle(DpyOfX, MainPixmap,
                       BgColorGC,
                       0, 0,
                       MainPixmapW,
                       MainPixmapH);
    }

    /* Layout Start */
    for(i = 0 ; i < LinesOfPage ; i++){
        sx = MainPixmapW -
             (FontDataInfo.box_w + FontDataInfo.ruby_box_w) * (i + 1);
        for(j = 0, ascii_half = 0 ; j < CharsOfLine ; j++){
            if(ptr == end){
                return end;
            }
            conv_failed = 0;
            if(ptr[0] == '\0'){
                return NULL;
            }
            if(ptr[0] == '\r'){
                ptr++;
                break;
            }
            if(ptr[0] == '\n'){
                ptr++;
                break;
            }

            sy = FontDataInfo.box_h * j;

            len = xtgvChrNum(ptr);
            if(len == 0){
                j--;
                if(j < -1){
                    j = -1;
                }
                ptr++;
                continue;
            }else
            if(len == 1){
                if(TagCutFlagOpt == 1){
                    r_ptr = xtgvHtmlTagDelete(ptr);
                    if(r_ptr != NULL){
                        use_line = xtgvHtmlTagCheck(ptr, sx, sy, flag);
                        if(use_line == 0){
                            /* Not Image Tag or Unable Draw Image */
                            ptr = r_ptr;
                            j--;
                            if(j <= -1){
                                /* Delete of tag only line */
                                if(ptr[0] == '\r'){
                                    ptr++;
                                }
                                if(ptr[0] == '\n'){
                                    ptr++;
                                }
                                j = -1;
                            }
                            continue;
                        }else
                        if(use_line == -1){
                            /* New Page */
                            return ptr;
                        }else{
                            /* Image Draw OK / New Line */
                            ptr = r_ptr;
                            i = i + use_line - 1;
                            if(j == 0){
                                /* Delete of tag only line */
                                if(ptr[0] == '\r'){
                                    ptr++;
                                }
                                if(ptr[0] == '\n'){
                                    ptr++;
                                }
                            }
                            break;
                        }
                    }
                }

                if(ascii_half == 0){
                    ascii_half = 1;
                }else{
                    j--;
                    if(j < -1){
                        j = -1;
                    }
                    sy = sy - FontDataInfo.box_h / 2;
                    ascii_half = 0;
                }

                if(flag == XTGV_DISPLAY_PAGE_ON){
                    ret = xtgvCreateXImage(&ascii_image,
                                           &wk_image,
                                           ptr, len);
                    if(ret != 0){
                        xtgvDrawString(ptr, len, sx, sy, 0);
                    }else{
                        for(cy = 0 ; cy < FontDataInfo.box_h ; cy++){
                            for(cx = 0 ; cx < (FontDataInfo.box_w / 2) ; cx++){
                                pixel = XGetPixel(wk_image, cx, cy);
                                XPutPixel(ascii_image,
                                          FontDataInfo.box_w - 1 - cy,
                                          cx,
                                          pixel);
                            }
                        }

                        if(wk_image != NULL){
                            XDestroyImage(wk_image);
                        }

                        XPutImage(DpyOfX, MainPixmap, FgColorGC,
                                  ascii_image,
                                  0, 0,
                                  sx, sy,
                                  FontDataInfo.box_w,
                                  FontDataInfo.box_h / 2);

                        if(ascii_image != NULL){
                            XDestroyImage(ascii_image);
                        }
                    }
                }
                ptr++;
            }else{
                /* len >= 2 */
#ifdef XXXXXXXXXXXXXXXX
                if(j == 0){
                    /* Line Top */
                    ret = xtgvKinsokuCheck(ptr);
                    if(ret == XTGV_NO_TOP_CHAR){
                    }
                }
#endif
                ascii_half = 0;
                if(flag == XTGV_DISPLAY_PAGE_ON){
                    wk_cce = xtgvConvTypeCheck(ptr);
                    if(wk_cce == NULL){
                        conv_type = XTGV_CONV_NONE;
                    }else{
                        conv_type = wk_cce->type;
                    }

                    switch(conv_type){
                    case XTGV_ROTATE_CW:
                    case XTGV_UP_RIGHT:
                        if(wk_cce->conv_image == NULL){
                            ret = xtgvCreateXImage(&(wk_cce->conv_image),
                                                   &wk_image,
                                                   ptr, len);
                            if(ret != 0){
                                conv_failed = 1;
                                break;
                            }

                            for(cy = 0 ; cy < FontDataInfo.box_h ; cy++){
                                for(cx = 0 ; cx < FontDataInfo.box_w ; cx++){
                                    pixel = XGetPixel(wk_image, cx, cy);
                                    if(conv_type == XTGV_ROTATE_CW){
                                        XPutPixel(wk_cce->conv_image,
                                                  FontDataInfo.box_w - 1 - cy,
                                                  cx,
                                                  pixel);
                                    }else
                                    if(conv_type == XTGV_UP_RIGHT){
                                        if(cx < (FontDataInfo.box_w / 2) &&
                                           cy >= (FontDataInfo.box_h / 2)){
                                            XPutPixel(wk_cce->conv_image,
                                                    cx + FontDataInfo.box_w / 2,
                                                    cy - FontDataInfo.box_h / 2,
                                                      pixel);
                                        }
                                    }
                                }
                            }

                            if(wk_image != NULL){
                                XDestroyImage(wk_image);
                            }
                        }

                        XPutImage(DpyOfX, MainPixmap, FgColorGC,
                                  wk_cce->conv_image,
                                  0, 0,
                                  sx, sy,
                                  FontDataInfo.box_w,
                                  FontDataInfo.box_h);
                        break;
                    case XTGV_MOVE_SMALL:
                        if(wk_cce->conv_image == NULL){
                            ret = xtgvCreateXImage(&(wk_cce->conv_image),
                                                   &wk_image,
                                                   ptr, len);
                            if(ret != 0){
                                conv_failed = 1;
                                break;
                            }

                            bg_sx = FontDataInfo.box_w;
                            bg_sy = -1;
                            for(cy = 0 ; cy < FontDataInfo.box_h ; cy++){
                                for(cx = 0 ; cx < FontDataInfo.box_w ; cx++){
                                    pixel = XGetPixel(wk_image, cx, cy);
                                    if(pixel != WinBgPixel){
                                        if(bg_sy == -1){
                                            bg_sy = cy;
                                        }
                                        if(bg_sx > cx){
                                            bg_sx = cx;
                                        }
                                    }
                                }
                            }

                            bg_ex = -1;
                            bg_ey = -1;
                            for(cy = FontDataInfo.box_h - 1 ; cy >= 0 ; cy--){
                                for(cx = FontDataInfo.box_w - 1 ;
                                    cx >= 0 ; cx--){
                                    pixel = XGetPixel(wk_image, cx, cy);
                                    if(pixel != WinBgPixel){
                                        if(bg_ey == -1){
                                            bg_ey = cy;
                                        }
                                        if(cx > bg_ex){
                                            bg_ex = cx;
                                        }
                                    }
                                }
                            }

                            bg_ex = bg_sx + bg_ex;
                            bg_sx = FontDataInfo.box_w - bg_ex;

                            bg_ey = FontDataInfo.box_h - bg_ey;
                            bg_sy = bg_sy - bg_ey;

                            if(bg_sx < 0 ||
                               bg_sx >= FontDataInfo.box_w){
                                bg_sx = 0;
                            }
                            if(bg_sy < 0 ||
                               bg_sy >= FontDataInfo.box_h){
                                bg_sy = 0;
                            }

                            for(cy = bg_sy, dy = 0 ;
                                cy < FontDataInfo.box_h ; cy++, dy++){
                                for(cx = 0, dx = bg_sx ;
                                    dx < FontDataInfo.box_w ;
                                    cx++, dx++){
                                    pixel = XGetPixel(wk_image, cx, cy);
                                    XPutPixel(wk_cce->conv_image,
                                              dx,
                                              dy,
                                              pixel);
                                }
                            }

                            if(wk_image != NULL){
                                XDestroyImage(wk_image);
                            }
                        }

                        XPutImage(DpyOfX, MainPixmap, FgColorGC,
                                  wk_cce->conv_image,
                                  0, 0,
                                  sx, sy,
                                  FontDataInfo.box_w,
                                  FontDataInfo.box_h);
                        break;
                    default:
                        conv_failed = 1;
                        break;
                    }

                    if(conv_failed == 1){
                        xtgvDrawString(ptr, len, sx, sy, 0);
                    }
                }
                ptr = ptr + len;
            }

            if(j == (CharsOfLine - 1)){
                if(ptr[0] == '\r'){
                    ptr++;
                    break;
                }
                if(ptr[0] == '\n'){
                    ptr++;
                    break;
                }
            }
        }

#ifdef XXXXXXXXXXXXXXXX
        /* Line End */
        ret = xtgvKinsokuCheck(ptr);
        if(ret == XTGV_NO_END_CHAR){
            if(flag == XTGV_DISPLAY_PAGE_ON){
                xtgvDeleteChar(sx, sy);
            }
        }
#endif
    }

    if(ptr[0] == '\0'){
        return NULL;
    }

    /* Layout End */
    return ptr;
}

int xtgvGotoPage(int page)
{
    unsigned char *next;

    if(page <= 0){
        for( ; PageDataList.complete != XTGV_TRUE ; ){
            if(xtgvAnalyzePage() != 0){
                return 1;
            }
        }
        if(page == 0){
            page = 1;
        }else{
            page = PageDataList.exist;
        }
    }

    if(PageDataList.current == page){
        return 0;
    }

    if(page > PageDataList.exist){
        for( ; PageDataList.complete != XTGV_TRUE ; ){
            if(xtgvAnalyzePage() != 0){
                return 1;
            }
            if(PageDataList.exist >= page){
                break;
            }
        }
    }

    if(PageDataList.complete == XTGV_TRUE){
        if(page > PageDataList.exist){
            page = PageDataList.exist;
        }
    }

    if(PageDataList.current == page){
        return 0;
    }

    next = xtgvLayoutPage(PageDataList.page_data[page - 1].start,
                          PageDataList.page_data[page - 1].next,
                          XTGV_DISPLAY_PAGE_ON);

    if(xtgvAnalyzePage() != 0){
        return 1;
    }

    PageDataList.current = page;

    xtgvPageReDraw();

    return 0;
}

int xtgvAnalyzePage(void)
{
    XtgvPDE *pde_wk;
    int i;
    unsigned char *ptr;
    unsigned char *next;

    if(PageDataList.complete == XTGV_TRUE){
        return 0;
    }

    if(PageDataList.page_data == NULL){
        PageDataList.page_data = (XtgvPDE *)malloc(sizeof(XtgvPDE) *
                                                   XTGV_PAGE_MAX);
        if(PageDataList.page_data == NULL){
            perror("malloc");
            return 1;
        }
        memset(PageDataList.page_data, 0, sizeof(XtgvPDE) * XTGV_PAGE_MAX);
        PageDataList.max = XTGV_PAGE_MAX;

        PageDataList.page_data[0].start = ReadData;
        PageDataList.page_data[0].next = NULL;
    }
    if(PageDataList.exist >= PageDataList.max){
        pde_wk = (XtgvPDE *)malloc(sizeof(XtgvPDE) *
                                   (PageDataList.max + XTGV_PAGE_MAX));
        if(pde_wk == NULL){
            perror("malloc");
            return 1;
        }
        memset(pde_wk, 0, sizeof(XtgvPDE) *
               (PageDataList.max + XTGV_PAGE_MAX));
        PageDataList.max = PageDataList.max + XTGV_PAGE_MAX;
        /* Copy */
        for(i = 0 ; i < PageDataList.exist ; i++){
            pde_wk[i].start = PageDataList.page_data[i].start;
            pde_wk[i].next = PageDataList.page_data[i].next;
        }

        free(PageDataList.page_data);
        PageDataList.page_data = pde_wk;
    }

    if(PageDataList.exist == 0){
        ptr = ReadData;
    }else{
        ptr = PageDataList.page_data[PageDataList.exist - 1].next;
        PageDataList.page_data[PageDataList.exist].start = ptr;
    }

    next = xtgvLayoutPage(ptr, NULL, XTGV_DISPLAY_PAGE_OFF);
    if(next == NULL){
        PageDataList.complete = XTGV_TRUE;
        PageDataList.exist++;
        return 0;
    }

    PageDataList.page_data[PageDataList.exist].next = next;

    PageDataList.exist++;

    return 0;
}

int xtgvGetFont(void)
{
    int  num_missing_charsets;
    char **missing_charsets;
    char *default_string;
    XRectangle dummy, bb;
#ifdef USE_XFT
    int  len;
    XftPattern *parse_pattern;
    XftPattern *match_pattern;
    XftResult match_result;
    XGlyphInfo gi;
    unsigned char dummy_char [4];
#endif /* USE_XFT */

    if(setlocale(LC_ALL, CurrentLang) == NULL){
        fprintf(stderr, "Set locale failed.\n");
        return 1;
    }

    if(XSupportsLocale() == 0){
        fprintf(stderr, "Not support locale.\n");
        return(0);
    }

#ifdef USE_XFT
    if(XftUseFlag == 1){
        if(CurrentLocale != XTGV_UTF){
            IconvCD = iconv_open("UTF-8", "EUC-JP");
            if(IconvCD == NULL){
                fprintf(stderr,"iconv open failed.\n");
                return(0);
            }
        }

        parse_pattern = XftNameParse(FontNameXft);
        if(parse_pattern == NULL){
            fprintf(stderr, "Font pattern parse failed.\n");
            return 1;
        }

        match_pattern = XftFontMatch(DpyOfX, XTGV_GET_SCREEN,
                                 parse_pattern, &match_result);
        if(match_pattern == NULL){
            fprintf(stderr, "Font pattern match failed.\n");
            XftPatternDestroy(parse_pattern);
            return 1;
        }

        XftPatternDestroy(parse_pattern);

        FontXft = XftFontOpenPattern(DpyOfX, match_pattern);
        if(FontXft == NULL){
            fprintf(stderr, "Font can't created.\n");
            XftPatternDestroy(match_pattern);
            return 1;
        }

        /* XftPatternDestroy is Segmentation fault at XCloseDisplay */
        //XftPatternDestroy(match_pattern);

        dummy_char [0] = 0xe3; //0xe2;
        dummy_char [1] = 0x82; //0x94;
        dummy_char [2] = 0x82; //0xbc;
        dummy_char [3] = 0x00;
        len = 3;

        XftTextExtentsUtf8(DpyOfX, FontXft, (FcChar8 *)dummy_char, len, &gi);

        FontDataInfo.char_w = (int)gi.xOff;
        FontDataInfo.char_h = (int)gi.xOff;

        FontDataInfo.ruby_char_w = FontDataInfo.char_w / 2;
        FontDataInfo.ruby_char_h = FontDataInfo.char_h / 2;
    }else{
#endif /* USE_XFT */
        FontSetXCF = XCreateFontSet(DpyOfX,
                                    FontNameXCF,
                                    &missing_charsets,
                                    &num_missing_charsets,
                                    &default_string);
        if(FontSetXCF == NULL){
            fprintf(stderr, "Font can't created.\n");
            return 1;
        }

        XmbTextExtents(FontSetXCF, "W", 1, &dummy, &bb);

        FontDataInfo.char_w = bb.width * 2;
        FontDataInfo.char_h = bb.height;

        FontDataInfo.ruby_char_w = FontDataInfo.char_w / 2;
        FontDataInfo.ruby_char_h = FontDataInfo.char_h / 2;
#ifdef USE_XFT
    }
#endif /* USE_XFT */

    //FontDataInfo.gap_x = FontDataInfo.char_w * 2 / 10;
    //FontDataInfo.gap_y = FontDataInfo.char_h * 2 / 10;
    FontDataInfo.gap_x = 1;
    FontDataInfo.gap_y = 1;

    //FontDataInfo.ruby_gap_x = FontDataInfo.ruby_char_w * 2 / 10;
    //FontDataInfo.ruby_gap_y = FontDataInfo.ruby_char_h * 2 / 10;
    FontDataInfo.ruby_gap_x = 0;
    FontDataInfo.ruby_gap_y = 0;

    FontDataInfo.ruby_box_w = FontDataInfo.ruby_char_w +
                              FontDataInfo.ruby_gap_x * 2;
    FontDataInfo.ruby_box_h = FontDataInfo.ruby_char_h +
                              FontDataInfo.ruby_gap_y * 2;

    FontDataInfo.box_w = FontDataInfo.char_w + FontDataInfo.gap_x * 2;
    FontDataInfo.box_h = FontDataInfo.char_h + FontDataInfo.gap_y * 2;

    return 0;
}

int xtgvReadCharConfig(char *path)
{
    FILE *fp;
    char conf_buf[XTGV_CHAR_CONFIG_LINE_MAX + 1];
    char *ptr;
    int  type;
    int  num;
    unsigned char conf_str[XTGV_CHAR_LEN];
    int  sx;
    int  sy;
    unsigned int conf_wk[XTGV_CHAR_LEN];
    int  i;
    int j;
    int  ret;
    int  dummy;
    XtgvCCE *cce_wk;

    fp = fopen(path, "r");
    if(fp == NULL){
        perror("fopen");
        return 1;
    }

    while(fgets(conf_buf, XTGV_CHAR_CONFIG_LINE_MAX, fp)){
        ptr = strrchr(conf_buf, '\n');
        if(ptr != NULL){
            ptr[0] = '\0';
        }
        ptr = strrchr(conf_buf, '\r');
        if(ptr != NULL){
            ptr[0] = '\0';
        }
        if(conf_buf[0] == '#'){
            continue;
        }
        if(strncmp(conf_buf, XTGV_LANG_LABEL, strlen(XTGV_LANG_LABEL)) == 0){
            ptr = conf_buf + strlen(XTGV_LANG_LABEL);
            if(ptr[0] != ' '){
                continue;
            }
            ptr++;

            if(strcmp(ptr, CurrentLang) != 0){
                fprintf(stderr, "config file is not match locale.\n");
                fclose(fp);
                return 2;
            }
            continue;
        }

        type = 0;
        num = 0;
        memset(conf_str, 0x00, XTGV_CHAR_LEN);
        sx = 0;
        sy = 0;
        ptr = conf_buf;
        memset(conf_wk, 0x00, XTGV_CHAR_LEN);

        ret = sscanf(ptr, "%d %d ", &type, &num);
        if(ret != 2){
            continue;
        }

        switch(num){
        case 1:
            ret = sscanf(ptr, "%d %d 0x%02x",
                         &dummy, &dummy,
                         &(conf_wk[0]));
            if(ret != (num + 2)){
                continue;
            }
            conf_str[0] = conf_wk[0];
            break;
        case 2:
            ret = sscanf(ptr, "%d %d 0x%02x 0x%02x",
                         &dummy, &dummy,
                         &(conf_wk[0]),
                         &(conf_wk[1]));
            if(ret != (num + 2)){
                continue;
            }
            conf_str[0] = conf_wk[0];
            conf_str[1] = conf_wk[1];
            break;
        case 3:
            ret = sscanf(ptr, "%d %d 0x%02x 0x%02x 0x%02x",
                         &dummy, &dummy,
                         &(conf_wk[0]),
                         &(conf_wk[1]),
                         &(conf_wk[2]));
            if(ret != (num + 2)){
                continue;
            }
            conf_str[0] = conf_wk[0];
            conf_str[1] = conf_wk[1];
            conf_str[2] = conf_wk[2];
            break;
        default:
            continue;
        }

        /* Add entry */
        if(ConvCharList.ptr == NULL){
            ConvCharList.ptr = (XtgvCCE *)malloc(sizeof(XtgvCCE) *
                                                 XTGV_CONV_CHAR_ENTRY_MAX);
            if(ConvCharList.ptr == NULL){
                perror("malloc");
                return 1;
            }
            memset(ConvCharList.ptr, 0,
                   sizeof(XtgvCCE) * XTGV_CONV_CHAR_ENTRY_MAX);
            ConvCharList.max = XTGV_CONV_CHAR_ENTRY_MAX;
        }

        if(ConvCharList.exist >= ConvCharList.max){
            cce_wk = (XtgvCCE *)malloc(sizeof(XtgvCCE) *
                                       (ConvCharList.max +
                                       XTGV_CONV_CHAR_ENTRY_MAX));
            if(cce_wk == NULL){
                perror("malloc");
                return 1;
            }
            memset(cce_wk, 0,
                   sizeof(XtgvCCE) *
                   (ConvCharList.max + XTGV_CONV_CHAR_ENTRY_MAX));
            ConvCharList.max = ConvCharList.max + XTGV_CONV_CHAR_ENTRY_MAX;
            /* Copy */
            for(i = 0 ; i < ConvCharList.exist ; i++){
                cce_wk[i].type = ConvCharList.ptr[i].type;
                cce_wk[i].num = ConvCharList.ptr[i].num;
                cce_wk[i].sx = ConvCharList.ptr[i].sx;
                cce_wk[i].sy = ConvCharList.ptr[i].sy;
                for(j = 0 ; j < XTGV_CHAR_LEN ; j++){
                    cce_wk[i].str[j] = ConvCharList.ptr[i].str[j];
                }
            }

            free(ConvCharList.ptr);
            ConvCharList.ptr = cce_wk;
        }

        ConvCharList.ptr[ConvCharList.exist].type = type;
        ConvCharList.ptr[ConvCharList.exist].num = num;
        ConvCharList.ptr[ConvCharList.exist].sx = sx;
        ConvCharList.ptr[ConvCharList.exist].sy = sy;
        for(i = 0 ; i < XTGV_CHAR_LEN ; i++){
            ConvCharList.ptr[ConvCharList.exist].str[i] = conf_str[i];
        }
        ConvCharList.ptr[ConvCharList.exist].conv_image = NULL;

        ConvCharList.exist++;
    }

    fclose(fp);
    return 0;
}

void xtgvWriteConfig(FILE *fp, unsigned char cl[][XTGV_CHAR_LEN], int type)
{
    int  i;
    int  j;
    int  num;

    for(i = 0 ; cl[i][0] != '\0' ; i++){
        num = xtgvChrNum(cl[i]);
        fprintf(fp, "%d %d ", type, num);
        for(j = 0 ; j < num ; j++){
            fprintf(fp, "%#02x ", cl[i][j]);
        }
        fprintf(fp, "# %s\n", cl[i]);
    }
    fprintf(fp, "\n");

    return;
}

/*
 * Create Config Dir       [~/.xtgv]
 * Create Char Config File [~/.xtgv/char.conf]
 * Read Char Config File   [~/.xtgv/char.conf]
 * Read Option Config File [~/.xtgv/option.conf]
 */
void xtgvCreateConfig(void)
{
    FILE *path_fp;
    int  path_fd;
    int  path_size;
    char *path_buf;
    struct stat path_stat;
    struct passwd *pwd_data;

    /* Get Home Dir */
    pwd_data = (struct passwd *)getpwuid(getuid());
    if(pwd_data == NULL){
        perror("getpwuid");
        exit(1);
    }

    path_size = strlen(pwd_data->pw_dir) + 1 +
                strlen(XTGV_CONF_DIR) + 1 +
                strlen(XTGV_CONF_CHAR_FILE) +
                strlen(XTGV_CONF_OPT_FILE);

    path_buf = (char *)malloc(path_size + 1);
    if(path_buf == NULL){
        perror("malloc");
        exit(1);
    }
    memset(path_buf, 0x00, path_size + 1);

    /* Set Config Dir Path */
    sprintf(path_buf, "%s/%s", pwd_data->pw_dir, XTGV_CONF_DIR);

    /* Check Config Dir */
    if(stat(path_buf, &path_stat) != 0){
        /* Create Config Dir */
        if(mkdir(path_buf , S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH) != 0){
            perror("mkdir");
            free(path_buf);
            exit(1);
        }
    }

    /* Set Char Config File Path */
    sprintf(path_buf, "%s/%s/%s",
            pwd_data->pw_dir, XTGV_CONF_DIR, XTGV_CONF_CHAR_FILE);

    /* Check Char Config File */
    if(stat(path_buf, &path_stat) != 0){
        /* Create Char Config File */
        path_fd = open(path_buf, O_CREAT|O_WRONLY|O_TRUNC,
                       S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
        if(path_fd == -1){
            perror("open");
            free(path_buf);
            exit(1);
        }
        path_fp = fdopen(path_fd, "w");
        if(path_fp == NULL){
            perror("fdopen");
            free(path_buf);
            close(path_fd);
            exit(1);
        }

        if(CurrentLocale == XTGV_UTF){
            /* Use UTF-8 */
            fprintf(path_fp, "%s %s\n\n", XTGV_LANG_LABEL, CurrentLang);

            xtgvWriteConfig(path_fp, rcw_char_utf, XTGV_ROTATE_CW);
            xtgvWriteConfig(path_fp, ur_char_utf, XTGV_UP_RIGHT);
            xtgvWriteConfig(path_fp, ms_char_utf, XTGV_MOVE_SMALL);
            xtgvWriteConfig(path_fp, uk_char_utf, XTGV_CONV_NONE);
        }else{
            /* Use EUC */
            fprintf(path_fp, "%s %s\n\n", XTGV_LANG_LABEL, CurrentLang);

            xtgvWriteConfig(path_fp, rcw_char_euc, XTGV_ROTATE_CW);
            xtgvWriteConfig(path_fp, ur_char_euc, XTGV_UP_RIGHT);
            xtgvWriteConfig(path_fp, ms_char_euc, XTGV_MOVE_SMALL);
            xtgvWriteConfig(path_fp, uk_char_euc, XTGV_CONV_NONE);
        }

        /* Close Char Config File */
        fclose(path_fp);
    }

    /* Read Char Config File */
    if(xtgvReadCharConfig(path_buf) != 0){
        fprintf(stderr, "[%s] read failed.\n", path_buf);
        free(path_buf);
        exit(1);
    }

    /* Set Option Config File Path */
    sprintf(path_buf, "%s/%s/%s",
            pwd_data->pw_dir, XTGV_CONF_DIR, XTGV_CONF_OPT_FILE);

    /* Read Option Config File */
    xtgvReadOptionConfig(path_buf);

    free(path_buf);
    return;
}

int xtgvChrNum(unsigned char *str)
{
    int num = 0;

    if(CurrentLocale == XTGV_UTF){
        /* Use UTF-8 */
        num = xtgvUTFCount(str[0]);
    }else{
        /* Use EUC */
        num = xtgvEUCCount(str[0]);
    }

    return num;
}

int xtgvEUCCount(unsigned char str)
{
    int num = 0;

    if(str < 0x1f || str == 0x7f){
        /* Control Code */
        num = 0;
    }else
    if(str <= 0x7f){
        /* 1 byte */
        num = 1;
    }else
    if(str <= 0x8e){
        /* hankaku katakana */
        num = 2;
    }else
    if(str <= 0x8f){
        /* hojyo kanji */
        num = 3;
    }else
    if(str >= 0xa1){
        /* kanji */
        num = 2;
    }else{
        /* None Use */
        num = 0;
    }

    return num;
}

int xtgvUTFCount(unsigned char str)
{
    int num = 0;

    if(str < 0x1f || str == 0x7f){
        /* Control Code */
        num = 0;
    }else
    if(str <= 0x7f){
        /* 1 byte */
        num = 1;
    }else
    if(str <= 0xbf){
        /* Continue */
        num = 0;
    }else
    if(str <= 0xdf){
        /* 2 byte */
        num = 2;
    }else
    if(str <= 0xef){
        /* 3 byte */
        num = 3;
    }else
    if(str <= 0xf7){
        /* 4 byte */
        num = 4;
    }else
    if(str <= 0xfb){
        /* 5 byte */
        num = 5;
    }else
    if(str <= 0xfd){
        /* 6 byte */
        num = 6;
    }else{
        /* None Use */
        num = 0;
    }

    return num;
}

int main(int argc, char **argv)
{
    int  ret;
    int  data_size;
    int  new_size;
    int  stdin_size;
    int  data_fd;
    unsigned char *new_buf;
    unsigned char *work_buf;
    struct stat data_stat;
    XSizeHints     xsh;
    XTextProperty   w_title_property;
    XTextProperty   i_title_property;
    int screen;
    XColor          fg_color[2];
    XColor          bg_color[2];
    unsigned char *title_name;

    stdin_size = 32 * 1024;

    xtgvInit();

    /* Basename Command */
    CmdName = strrchr(argv[0], '/');
    if(CmdName == NULL){
        CmdName = argv[0];
    }else{
        CmdName++;
    }

    /* Get Env LANG */
    CurrentLang = getenv(XTGV_LANG_LABEL);
    if(CurrentLang == NULL){
        fprintf(stderr, "LANG not set.\n");
        exit(1);
    }

    if(strcmp(CurrentLang, XTGV_LANG_UTF) == 0){
        CurrentLocale = XTGV_UTF;
    }else
    if(strcmp(CurrentLang, XTGV_LANG_EUC) == 0){
        CurrentLocale = XTGV_EUC;
    }else{
        fprintf(stderr, "Not support locale.\n");
        exit(1);
    }

    xtgvCreateConfig();

    if(xtgvCheckOption(argc, argv, 1) != 0){
        exit(1);
    }

#ifdef USE_XFT
    if(XftUseFlag == 1){
        if(FontNameOpt != NULL){
            FontNameXft = FontNameOpt;
        }else
        if(FontNameOptConf[0] != '\0'){
            FontNameXft = FontNameOptConf;
        }
    }else{
#endif /* USE_XFT */
        if(FontNameOpt != NULL){
            FontNameXCF = FontNameOpt;
        }else
        if(FontNameOptConf[0] != '\0'){
            FontNameXCF = FontNameOptConf;
        }
#ifdef USE_XFT
    }
#endif /* USE_XFT */

    if(FgColorNameOpt != NULL){
        FgColorName = FgColorNameOpt;
    }else
    if(FgColorNameOptConf[0] != '\0'){
        FgColorName = FgColorNameOptConf;
    }

    if(BgColorNameOpt != NULL){
        BgColorName = BgColorNameOpt;
    }else
    if(BgColorNameOptConf[0] != '\0'){
        BgColorName = BgColorNameOptConf;
    }

    if(LinesOfPageOpt != 0){
        LinesOfPage = LinesOfPageOpt;
    }

    if(CharsOfLineOpt != 0){
        CharsOfLine = CharsOfLineOpt;
    }

    if(FileNameOpt == NULL){
        /* Read Stdin */
        work_buf = (unsigned char *)malloc(stdin_size + 1);
        if(work_buf == NULL){
            perror("malloc");
            exit(1);
        }
        memset(work_buf, 0x00, stdin_size + 1);

        for(data_size = 0, new_size = 0 ; ; ){
            ret = read(STDIN_FILENO, work_buf, stdin_size);
            if(ret < 0){
                perror("read");
                free(work_buf);
                exit(1);
            }else if(ret == 0){
                /* EOF */
                break;
            }else{
                new_size = data_size + ret;
                new_buf = (unsigned char *)malloc(new_size + 1);
                if(new_buf == NULL){
                    perror("malloc");
                    free(work_buf);
                    if(ReadData != NULL){
                        free(ReadData);
                    }
                    exit(1);
                }
                memset(new_buf, 0x00, new_size + 1);
                if(ReadData != NULL){
                    memcpy(new_buf, ReadData, data_size);
                    free(ReadData);
                }
                memcpy(new_buf + data_size, work_buf, ret);
                ReadData = new_buf;
                data_size = new_size;
                new_buf = NULL;
                new_size = 0;
            }
        }

        free(work_buf);
    }else{
        /* Check File */
        if(stat(FileNameOpt, &data_stat) != 0){
            perror("stat");
            exit(1);
        }
        data_size = data_stat.st_size;

        /* Open File */
        data_fd = open(FileNameOpt, O_RDONLY);
        if(data_fd == -1){
            perror("open");
            exit(1);
        }

        ReadData = (unsigned char *)malloc(data_size + 1);
        if(ReadData == NULL){
            perror("malloc");
            exit(1);
        }
        memset(ReadData, 0x00, data_size + 1);

        /* Read File */
        if(read(data_fd, ReadData, data_size) < 0){
            perror("read");
            free(ReadData);
            exit(1);
        }

        /* Close File */
        close(data_fd);
        data_fd = -1;
    }

    if(data_size == 0){
        fprintf(stderr, "read data size is zero.\n");
        if(ReadData != NULL){
            free(ReadData);
        }
        exit(1);
    }

    DpyOfX = XOpenDisplay(NULL);
    if(DpyOfX == NULL){
        fprintf(stderr, "Display can't be opened.\n");
        exit(1);
    }

    screen = DefaultScreen(DpyOfX);

#if defined(USE_IMLIB2)
    imlib_context_set_display(DpyOfX);
    imlib_context_set_visual(XTGV_GET_VISUAL);
    imlib_context_set_colormap(XTGV_GET_CMAP);
#elif defined(USE_IMLIB)
    ImlibDataExt = Imlib_init(DpyOfX);

    if(ImlibDataExt->x.depth != DefaultDepth(DpyOfX, screen)){
        ImlibDataExt->x.depth = DefaultDepth(DpyOfX, screen);
        ImlibDataExt->x.render_depth = ImlibDataExt->x.depth;
    }
#endif /* USE_IMLIB2 */

    /* Get Font */
    if(xtgvGetFont() != 0){
        XCloseDisplay(DpyOfX);
        exit(1);
    }

    MainPixmapW = FontDataInfo.ruby_box_w +
               (FontDataInfo.box_w + FontDataInfo.ruby_box_w) *
               LinesOfPage;
    /* + 2 for kinsoku */
    MainPixmapH = FontDataInfo.box_h * (CharsOfLine + 2);

    MainWinW = MainPixmapW;
    MainWinH = MainPixmapH + FontDataInfo.box_h + FontDataInfo.box_h / 2;

    ret = XAllocNamedColor(DpyOfX, XTGV_GET_CMAP,
                              FgColorName, &fg_color[0], &fg_color[1]);
    if(ret == 0){
        fprintf(stderr, "Worning Get Color Use Black.\n");
        fg_color[0].pixel = BlackPixel(DpyOfX, screen);
    }

    ret = XAllocNamedColor(DpyOfX, XTGV_GET_CMAP,
                              BgColorName, &bg_color[0], &bg_color[1]);
    if(ret == 0){
        fprintf(stderr, "Worning Get Color Use White.\n");
        bg_color[0].pixel = WhitePixel(DpyOfX, screen);
    }

    WinFgPixel = fg_color[0].pixel;
    WinBgPixel = bg_color[0].pixel;

    /* Create Window */
    MainWin = XCreateSimpleWindow(DpyOfX, RootWindow(DpyOfX, screen),
                                  0, 0,
                                  MainWinW, MainWinH,
                                  1,
                                  WinFgPixel,
                                  WinBgPixel);

    xsh.flags  = (PBaseSize | USSize | PMinSize | PMaxSize);
    xsh.width       = MainWinW;
    xsh.height      = MainWinH;
    xsh.base_width  = MainWinW;
    xsh.base_height = MainWinH;
    xsh.min_width   = MainWinW;
    xsh.min_height  = MainWinH;
    xsh.max_width   = MainWinW;
    xsh.max_height  = MainWinH;

    XSetWMNormalHints(DpyOfX, MainWin, &xsh);

    /* Window Title */
    title_name = (unsigned char *)malloc(strlen(CmdName) +
                                         strlen(XtgvVersion) + 2);
    if(title_name == NULL){
        xtgvExit();
        exit(1);
    }
    memset(title_name, 0x00, strlen(CmdName) + strlen(XtgvVersion) + 2);

    sprintf((char *)title_name, "%s %s", CmdName, XtgvVersion);

    if(XmbTextListToTextProperty(DpyOfX,
                                 (char **)&title_name,
                                 1,
                                 XStdICCTextStyle,
                                 &w_title_property) >= Success){
        XSetWMName(DpyOfX, MainWin, &w_title_property) ;
        XFree(w_title_property.value) ;
    }else{
        XStoreName(DpyOfX, MainWin, CmdName) ;
    }

    free(title_name);

    if(XmbTextListToTextProperty(DpyOfX,
                                 (char **)&CmdName,
                                 1,
                                 XStdICCTextStyle,
                                 &i_title_property) >= Success){
        XSetWMIconName(DpyOfX, MainWin, &i_title_property) ;
        XFree(i_title_property.value) ;
    }else{
        XSetIconName(DpyOfX, MainWin, CmdName) ;
    }

    /* Set Check Event */
    XSelectInput(DpyOfX, MainWin, KeyPressMask |
                                  VisibilityChangeMask |
                                  ExposureMask);

    MainPixmap = XCreatePixmap(DpyOfX,
                               DefaultRootWindow(DpyOfX),
                               MainPixmapW,
                               MainPixmapH,
                               DefaultDepth(DpyOfX, screen));
    if(MainPixmap <= 0){
        xtgvExit();
        exit(1);
    }

    ConvPixmap = XCreatePixmap(DpyOfX,
                               DefaultRootWindow(DpyOfX),
                               FontDataInfo.box_w,
                               FontDataInfo.box_h,
                               DefaultDepth(DpyOfX, screen));
    if(ConvPixmap <= 0){
        xtgvExit();
        exit(1);
    }

#ifdef USE_XFT
    if(XftUseFlag == 1){
        XftColorAllocName(DpyOfX,
                          XTGV_GET_VISUAL, XTGV_GET_CMAP,
                          FgColorName, &ColorXft);

        DrawXft = XftDrawCreate(DpyOfX, MainPixmap,
                                XTGV_GET_VISUAL, XTGV_GET_CMAP);
        if(DrawXft == NULL){
            xtgvExit();
            exit(1);
        }

        ConvXft = XftDrawCreate(DpyOfX, ConvPixmap,
                                XTGV_GET_VISUAL, XTGV_GET_CMAP);
        if(ConvXft == NULL){
            xtgvExit();
            exit(1);
        }

        MainWinXft = XftDrawCreate(DpyOfX, MainWin,
                                   XTGV_GET_VISUAL, XTGV_GET_CMAP);
        if(MainWinXft == NULL){
            xtgvExit();
            exit(1);
        }
    }
#endif /* USE_XFT */

    FgColorGC = XCreateGC(DpyOfX, MainWin, 0, 0);
    if(FgColorGC <= 0){
        xtgvExit();
        exit(1);
    }
    XSetForeground(DpyOfX, FgColorGC, WinFgPixel);
    XSetBackground(DpyOfX, FgColorGC, WinBgPixel);

    BgColorGC = XCreateGC(DpyOfX, MainWin, 0, 0);
    if(BgColorGC <= 0){
        xtgvExit();
        exit(1);
    }
    XSetForeground(DpyOfX, BgColorGC, WinBgPixel);
    XSetBackground(DpyOfX, BgColorGC, WinFgPixel);

    /* Layout of Page */
    if(xtgvGotoPage(CurrentPageOpt) != 0){
        xtgvExit();
        exit(1);
    }

    /* Map Window */
    XMapWindow(DpyOfX, MainWin);

    XSync(DpyOfX, 0);

    /* Event Loop */
    xtgvEventLoop();

    xtgvExit();

    exit(0);
}

